<?php
namespace WPCS\Migration\API;

use WP_REST_Request;
use WPCS\API\CreateTenantRequest;
use WPCS\API\CreateVersionRequest;
use WPCS\API\GetVersionsRequest;
use WPCS\Migration\Export\Exporter;
use WPCS\Migration\Helpers;

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) )
    exit;

class Module
{
    public static function init()
    {
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    public static function register_routes()
    {
        $namespace = WPCS_MIGRATION_API_NAMESPACE . '/v1';

        register_rest_route($namespace, '/config/auth', [
            'methods' => \WP_REST_Server::EDITABLE,
            'callback' => [__CLASS__, 'handle_config_auth'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/config/auth/check', [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [__CLASS__, 'handle_config_auth_check'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/info/versions', [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [__CLASS__, 'handle_get_wpcs_info'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/info/dir-size', [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [__CLASS__, 'handle_get_dir_sizes'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/versions', [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [__CLASS__, 'handle_get_versions'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/migrate/tenant', [
            'methods' => \WP_REST_Server::CREATABLE,
            'callback' => [__CLASS__, 'handle_tenant_migration'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/migrate/version', [
            'methods' => \WP_REST_Server::CREATABLE,
            'callback' => [__CLASS__, 'handle_version_migration'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);

        register_rest_route($namespace, '/jobs/(?P<id>\w+)/status', [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [__CLASS__, 'handle_get_job_status'],
            'permission_callback' => [__CLASS__, 'permission_callback'],
        ]);
    }

    public static function permission_callback( WP_REST_Request $request)
    {
        return wp_verify_nonce($request->get_header('X-WP-Nonce'), 'wp_rest');
    }

    public static function handle_config_auth(WP_REST_Request $request)
    {
        $body = json_decode($request->get_body());

        update_option('WPCS_API_REGION', $body->region);
        update_option('WPCS_API_KEY', $body->key);
        update_option('WPCS_API_SECRET', $body->secret);

        return new \WP_REST_Response([
            'message' => 'success'
        ], 200);
    }

    public static function handle_config_auth_check(WP_REST_Request $request)
    {
        try {
            (new GetVersionsRequest())
                ->setRegion(Helpers::get_auth_value('WPCS_API_REGION'))
                ->setApiKey(Helpers::get_auth_value('WPCS_API_KEY'))
                ->setApiSecret(Helpers::get_auth_value('WPCS_API_SECRET'))
                ->send();
        } catch (\Throwable $th) {
            update_option('WPCS_IS_AUTHENTICATED', false);

            return new \WP_REST_Response([
                'message' => 'Credentials not valid.'
            ], 401);
        }

        update_option('WPCS_IS_AUTHENTICATED', true);

        return new \WP_REST_Response([
            'message' => 'success'
        ], 200);
    }

    public static function handle_get_versions()
    {
        $wpcs_request = new GetVersionsRequest();
        $response = $wpcs_request
            ->setRegion(Helpers::get_auth_value('WPCS_API_REGION'))
            ->setApiKey(Helpers::get_auth_value('WPCS_API_KEY'))
            ->setApiSecret(Helpers::get_auth_value('WPCS_API_SECRET'))
            ->send();

        $wpcs_ready_versions = array_filter($response, function($version) {
            return $version->statusCode === 1;
        });

        $wpcs_versions = array_map(function($version) {
            return [
                'id' => $version->id,
                'name' => $version->name,
                'isProduction' => $version->isProduction,
            ];
        }, $wpcs_ready_versions);

        return new \WP_REST_Response($wpcs_versions, 200);
    }

    public static function handle_get_wpcs_info()
    {
        global $wp_version;

        $url = 'https://s3.eu-central-1.amazonaws.com/wpcs.global.info/global/info.json';
        $response = wp_remote_get($url);

        if(is_wp_error($response))
        {
            return new \WP_REST_Response([
                "error" => "Uknown error"
            ], 500);
        }

        $body = json_decode($response['body']);

        $wp_versions = $body->wp;
        $php_versions = $body->php;
        $current_php_version = sprintf('%s.%s', PHP_MAJOR_VERSION, PHP_MINOR_VERSION);
        
        $return_body = [
            'wp' => [
                'versions' => $wp_versions,
                'recommended' => Helpers::find_closest_version($wp_version, $body->wp),
            ],
            'php' => [
                'versions' => $php_versions,
                'recommended' => Helpers::find_closest_version($current_php_version, $body->php),
            ],
        ];

        return new \WP_REST_Response($return_body, 200);
    }

    public static function handle_get_dir_sizes(WP_REST_Request $request)
    {
        $uploads_dir_size = Helpers::get_directory_size(wp_upload_dir()['basedir']);

        return new \WP_REST_Response([
            'message' => 'success',
            'uploads' => [
                'size' => $uploads_dir_size,
            ],
        ], 200);
    }

    public static function handle_tenant_migration(WP_REST_Request $request)
    {
        $old_max_execution_timout = ini_get('max_execution_time');
        ini_set('max_execution_time', 1800);

        $body = json_decode($request->get_body());

        $job_service = new JobService($body->jobId);
        $job_service->set_status('Exporting...');

        $exporter = new Exporter();
        $exporter
            ->setProgressCallback(function ($total_files, $files_packaged) use ($job_service) {
                $job_service->set_progress($total_files, $files_packaged);
            })
            ->setClosingZipCallback(function () use ($job_service) {
                $job_service->set_status('Finalizing export...');
            });

        $path = $exporter->create_package();
        
        $job_service->set_status('Uploading to WPCS...');
        
        try {
            $wpcs_request = new CreateTenantRequest();
            $wpcs_request
                ->setRegion(Helpers::get_auth_value('WPCS_API_REGION'))
                ->setApiKey(Helpers::get_auth_value('WPCS_API_KEY'))
                ->setApiSecret(Helpers::get_auth_value('WPCS_API_SECRET'))
                ->setName($body->name)
                ->setSnapshotPath($path)
                ->setUploadProgressCallback(function ($total, $now) use ($job_service) {
                    $job_service->set_progress($total, $now);
                })
                ->setExternalId($body->externalId);

            if($body->versionId)
            {
                $wpcs_request->setVersionId($body->versionId);
            }
            
            $response = $wpcs_request->send();
    
            $job_service->set_status('Done');

            return new \WP_REST_Response($response, 200);
        } catch (\Exception $exc) {
            $job_service->set_status('Error');

            return new \WP_REST_Response([
                'message' => $exc->getMessage(),
            ], $exc->getCode());
        } finally {
            unlink($path);
            $job_service->cleanup();
        }

        ini_set('max_execution_time', $old_max_execution_timout);
    }

    public static function handle_version_migration(WP_REST_Request $request)
    {
        $old_max_execution_timout = ini_get('max_execution_time');
        ini_set('max_execution_time', 1800);

        $body = json_decode($request->get_body());

        $job_service = new JobService($body->jobId);
        $job_service->set_status('Exporting...');
        
        $exporter = new Exporter();
        $exporter
            ->setProgressCallback(function ($total_files, $files_packaged) use ($job_service) {
                $job_service->set_progress($total_files, $files_packaged);
            })
            ->setClosingZipCallback(function () use ($job_service) {
                $job_service->set_status('Finalizing export...');
            });

        $path = $exporter->create_package([
            'skip_uploads' => $body->skipUploads ?? false,
        ]);

        $job_service->set_status('Uploading to WPCS...');

        try {
            $wpcs_request = new CreateVersionRequest();
            $response = $wpcs_request
                ->setRegion(Helpers::get_auth_value('WPCS_API_REGION'))
                ->setApiKey(Helpers::get_auth_value('WPCS_API_KEY'))
                ->setApiSecret(Helpers::get_auth_value('WPCS_API_SECRET'))
                ->setName($body->name)
                ->setSnapshotPath($path)
                ->setUploadProgressCallback(function ($total, $now) use ($job_service) {
                    $job_service->set_progress($total, $now);
                })
                ->setPhpVersion($body->phpVersion)
                ->setWordPressVersion($body->wordpressVersion)
                ->send();
    
            $job_service->set_status('Done');
    
            return new \WP_REST_Response($response, 200);
        } catch (\Exception $exc) {
            $job_service->set_status('Error');

            return new \WP_REST_Response([
                'message' => $exc->getMessage(),
            ], $exc->getCode());
        } finally {
            unlink($path);
            $job_service->cleanup();
        }

        ini_set('max_execution_time', $old_max_execution_timout);
    }

    public static function handle_get_job_status(WP_REST_Request $request)
    {
        $job_service = new JobService($request->get_param('id'));
        $status = $job_service->get_status();
        return new \WP_REST_Response($status, 200);
    }
}
