<?php
namespace WPCS\Migration\Export;

use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use WPCS\Migration\Helpers;
use ZipArchive;

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) )
    exit;

class Exporter
{
    private $progress_callback;
    private $closing_zip_callback;
    private $total_files;
    private $files_packaged;

    public function setProgressCallback($progress_callback)
    {
        $this->progress_callback = $progress_callback;

        return $this;
    }

    private function report_progress($total, $progress)
    {
        call_user_func($this->progress_callback, $total, $progress);
    }

    public function setClosingZipCallback($cb)
    {
        $this->closing_zip_callback = $cb;

        return $this;
    }

    public function create_package($options = [])
    {
        $rand = rand();
        $path = Helpers::join_paths(
            Helpers::get_wpcs_dir(),
            "site-{$rand}.zip",
        );

        $default_options = [
            'skip_uploads' => false,
        ];

        $this->create_snapshot($path, array_merge($default_options, $options));

        return $path;
    }

    public static function get_metadata($metadata = [])
    {
        return json_encode(array_merge($metadata, [
            'version' => 1,
            'originalDomainName' => str_replace(['http://', 'https://'], '', get_option('home')),
        ]));
    }
    
    public function create_snapshot($path, $options, $root_dir = 'var/www/html')
    {
        $zip = new ZipArchive();
        if ($zip->open($path, ZipArchive::CREATE) !== TRUE) {
            throw new \Exception("Could not open ZIP file at $path.");
        }
        
        // Add metadata file
        $metadata_path = Helpers::join_paths($root_dir, 'export', 'metadata.json');
        $zip->addFromString($metadata_path, self::get_metadata());

        // Add db
        $zipped_db_path = Helpers::join_paths($root_dir, 'export', 'db.sql');
        $db_path = Db::createDump(Helpers::get_wpcs_dir());
        $zip->addFile($db_path, $zipped_db_path);

        // Add wp-content
        $dir_iterator = new RecursiveDirectoryIterator(
            WP_CONTENT_DIR,
            RecursiveDirectoryIterator::FOLLOW_SYMLINKS,
        );
        $files = new RecursiveIteratorIterator($dir_iterator);

        $this->total_files = iterator_count($files);

        $excluded_files = [
            '.',
            '..',
            '.DS_Store',
        ];

        $excluded_paths = [
            is_string(WP_DEBUG_LOG) ? realpath(WP_DEBUG_LOG) : WP_CONTENT_DIR . '/debug.log',
        ];

        $excluded_dirs = [
            Helpers::get_wpcs_dir(false),
            WP_CONTENT_DIR . '/uploads/prime-mover-export-files',
            WP_CONTENT_DIR . '/ai1wm-backups',
        ];

        if($options['skip_uploads'])
        {
            $excluded_dirs[] = Helpers::join_paths(WP_CONTENT_DIR, 'uploads');
        }

        $files_percent = (int) $this->total_files/30;
        $i = 0;

        foreach ($files as $path => $file)
        {
            $this->files_packaged = $this->files_packaged + 1;

            if($file->getType() !== 'file') 
            {
                continue;
            }

            // Exclude the specified excludes
            if(in_array($file->getBasename(), $excluded_files) || in_array($path, $excluded_paths))
            {
                continue;
            }

            // Do not add anything in the excluded dirs
            if(Helpers::is_in_dirs($path, $excluded_dirs))
            {
                continue;
            }

            $relative_path = str_replace(ABSPATH, '', $path);
            if(DIRECTORY_SEPARATOR == '\\') 
            {
                $relative_path = str_replace(DIRECTORY_SEPARATOR,'/', $relative_path);
            }

            $zipped_path = Helpers::join_paths($root_dir, $relative_path);
            $zip->addFile($file->getRealPath(), $zipped_path);

            if(($i % $files_percent) == 0)
            {
                $this->report_progress($this->total_files, $this->files_packaged);
            }
            $i++;
        }

        // Progress means the progress of closing the ZIP. Only available in PHP >=8.0
        if(method_exists($zip, 'registerProgressCallback'))
        {
            $zip->registerProgressCallback(0.05, function ($rate) {
                $this->report_progress(1, $rate);
            });
        }
        else
        {
            // Setting total to zero makes frontend show no progress indication.
            $this->report_progress(0, 0);
        }

        call_user_func($this->closing_zip_callback);
        $zip->close();

        // Clean up
        unlink($db_path);
    }
}
