<?php

namespace WPCS\Migration;

use FilesystemIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;

class Helpers
{
    /**
     * Returns the version 'closest' to the provided version, from a list of provided versions.
     * 
     * Closest here means the version that matches exactly or the version that is just above the provided needle. When giving an array that includes '5.8.3' and '5.9', searching for '5.8.4' will yield '5.9'.
     *
     * @param string $version
     * @param string[] $version_array
     * @return string
     */
    public static function find_closest_version($version, $version_array)
    {
        usort($version_array, function($a, $b){
            return version_compare($a, $b);
        });

        foreach($version_array as $v)
        {
            $result = version_compare($v, $version);

            if($result === -1)
            {
                continue;
            }

            return $v;
        }

        // No match found, return highest (the last) version.
        return $version_array[ count($version_array) - 1 ];
    }

    /**
     * Join path components into a functional path
     *
     * @return string
     */
    public static function join_paths()
    {
        $paths = array();
    
        foreach (func_get_args() as $arg) {
            if ($arg !== '') { $paths[] = $arg; }
        }
    
        return preg_replace('#/+#','/', join('/', $paths));
    }

    /**
     * Get the path to the WPCS directory
     * 
     * Will also create the path if instructed to do so with the $create_if_not_exists param.
     *
     * @param boolean $create_if_not_exists Whether or not to create the path if it does not exist. Defaults to true.
     * @return string
     */
    public static function get_wpcs_dir($create_if_not_exists = true)
    {
        $wpcs_dir = self::join_paths(WP_CONTENT_DIR, 'wpcs');
        if(!is_dir($wpcs_dir) && $create_if_not_exists)
        {
            mkdir($wpcs_dir);
        }

        return $wpcs_dir;
    }

    /**
     * Checks if the given filepath is in the given directory
     *
     * @param string $filepath
     * @param string $dir
     * @return boolean
     */
    public static function is_in_dir($filepath, $dir)
    {
        if(!function_exists('str_starts_with'))
        {
            return substr($filepath, 0, strlen($dir)) === $dir;
        }

        return str_starts_with($filepath, $dir);
    }

    /**
     * Checks if the given filepath is in the given directory
     *
     * @param string $filepath
     * @param array $dir
     * @return boolean
     */
    public static function is_in_dirs($filepath, array $dirs)
    {
        foreach ($dirs as $dir) {
            $result = Helpers::is_in_dir($filepath, $dir);
            if ($result) {
                return $result;
            }
        }

        return false;
    }

    public static function is_authenticated()
    {
        try {
            Helpers::get_auth_value('WPCS_API_REGION');        
            Helpers::get_auth_value('WPCS_API_KEY');
            Helpers::get_auth_value('WPCS_API_SECRET');
        } catch (\Exception $exc) {
            update_option('WPCS_IS_AUTHENTICATED', false);
        }

        return get_option('WPCS_IS_AUTHENTICATED', false);
    }

    public static function get_auth_value($value_name)
    {
        if(defined($value_name))
        {
            return constant($value_name);
        }

        $option = get_option($value_name);
        if(!$option)
        {
            throw new \Exception("Could not find authentication value $value_name.");
        }

        return $option;
    }
    /**
     * Gets the size of a directory in bytes.
     *
     * @param string $path
     * @return int
     */
    public static function get_directory_size($path){
        $bytestotal = 0;
        $path = realpath($path);
        if($path!==false && $path!='' && file_exists($path)){
            foreach(new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS)) as $object){
                $bytestotal += $object->getSize();
            }
        }
        return $bytestotal;
    }
}
