<?php

namespace WPFunnels\Rest\Controllers;

use Error;
use WP_Error;
use WP_REST_Request;
use WPFunnels\Data_Store\Wpfnl_Steps_Store_Data;
use WPFunnels\Wpfnl_functions;
use WPFunnelsPro\AbTesting\Wpfnl_Ab_Testing;

use WPFunnels\Admin\Module\Steps\Wpfnl_Steps_Factory;
use WPFunnels\Admin\Module\Wpfnl_Admin_Module;
use WPFunnels\Wpfnl;


class AbTestingController extends Wpfnl_REST_Controller
{

    /**
     * Endpoint namespace.
     *
     * @var string
     */
    protected $namespace = 'wpfunnels/v1';

    /**
     * Route base.
     *
     * @var string
     */
    protected $rest_base = 'abtesting';

    /**
     * check if user has valid permission
     *
     * @param $request
     * @return bool|WP_Error
     * @since 1.0.0
     */
    public function update_items_permissions_check()
    {   
        
        if ( !Wpfnl_functions::wpfnl_rest_check_manager_permissions( 'steps', 'edit' ) ) {
            return new WP_Error('wpfunnels_rest_cannot_edit', __('Sorry, you cannot edit this resource.', 'wpfnl'), ['status' => rest_authorization_required_code()]);
        }
        return true;
    }

    /**
     * Makes sure the current user has access to READ the settings APIs.
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|boolean
     * @since  3.0.0
     */
    public function get_items_permissions_check( $request ) {
        return true;
        if( !Wpfnl_functions::wpfnl_rest_check_manager_permissions( 'settings' ) ) {
            return new WP_Error( 'wpfunnels_rest_cannot_view', __( 'Sorry, you cannot list resources.', 'wpfnl' ), [ 'status' => rest_authorization_required_code() ] );
        }
        return true;
    }


    /**
     * register rest routes
     *
     * @since 1.0.0
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->rest_base . '/update-status/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'update_status'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/create-varient/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'create_varient'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/duplicate-varient/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'duplicate_varient'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);
        
        register_rest_route($this->namespace, '/' . $this->rest_base . '/restore-varient/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'restore_varient'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/update-start-settings/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'update_start_settings'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/delete-variant/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'delete_variant'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/delete-archive/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'delete_archive'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/get-start-settings/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [
                    $this,
                    'get_start_settings'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/get-settings/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [
                    $this,
                    'get_settings'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/get-stats/(?P<funnel_id>[\d]+)/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [
                    $this,
                    'get_stats'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/reset-stats/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'reset_stats'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/reset-settings/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'reset_settings'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/declear-winner/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'declear_winner'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/archive-variant/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'archive_variant'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/get-winner/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [
                    $this,
                    'get_winner'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/stop-ab-testing/(?P<step_id>[\d]+)', [
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [
                    $this,
                    'stop_ab_testing'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ],
        ]);
    }


    /**
     * Update A/B testing status
     * 
     * @param WP_REST_Request $request
     * @return Array
     * 
     * @since 1.6.21
     */
    public function update_status( WP_REST_Request $request ){

        if( isset($request['step_id']) ){
            $step_id = $request['step_id'];
            $value   = isset($request['value']) ? $request['value'] : '';

            $response = Wpfnl_Ab_Testing::update_ab_testing_status( $step_id, $value );
          
            if( $response ){
                return rest_ensure_response( $this->get_success_response() );
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * Get setttings
     * 
     * @param WP_REST_Request $request
     * @return Array
     * 
     * @since 1.6.21
     */
    public function get_settings( WP_REST_Request $request ){
        
        
        $response = [];
        if( isset( $request['step_id'] ) ){
            $step_id = $request['step_id'];
            $default_settings = Wpfnl_Ab_Testing::get_default_start_setting( $step_id );
            //check A/B testing is enable or not
            
            $result = Wpfnl_Ab_Testing::maybe_ab_testing( $step_id );
            
            $default_settings['is_ab_enabled'] = $result ? $result : '';
           
            // get start settings
            $result = Wpfnl_Ab_Testing::get_start_settings( $step_id );
            $default_settings['start_settings'] = $result ? $result : $default_settings['start_settings'];

            $response = $this->get_success_response();
            if( isset($default_settings['start_settings']['variations']) && is_array($default_settings['start_settings']['variations']) ){
                foreach( $default_settings['start_settings']['variations'] as $key => $variation ){
                    if( isset($variation['id'], $variation['step_type']) && ( 'checkout' ==  $variation['step_type'] || 'upsell' ==  $variation['step_type'] || 'downsell' ==  $variation['step_type'])){
                        $step_products = get_post_meta( $variation['id'], '_wpfnl_'.$variation['step_type'].'_products', true );
                        if( is_array($step_products) && count($step_products) ){
                            $default_settings['start_settings']['variations'][$key]['is_product'] = 'yes'; 
                        }

                        if( 'checkout' ==  $variation['step_type'] ){
                            $ob_products = get_post_meta( $variation['id'], 'order-bump-settings', true );
                            if( is_array($ob_products) && count($ob_products) ){
                                $default_settings['start_settings']['variations'][$key]['is_ob'] = 'yes'; 
                            }
                        }
                    }
                }
            }
            $response['data']        = $default_settings;
            $response['step_type']  =  get_post_meta( $step_id, '_step_type', true );
          
        }else{
            $response = $this->get_error_response();
            $response['data']       = '';
            $response['step_type']  = '';
        }
        
        return rest_ensure_response( $response );
    }


    /**
     * Update/save start settings
     * 
     * @param WP_REST_Request $request
     * @return Array 
     * @since 1.6.21
     */
    public function update_start_settings( WP_REST_Request $request ){
        if( isset($request['step_id']) && isset($request['data']) ){
           
            $step_id = $request['step_id'];
            $data    = $request['data'];
            if( isset($data['auto_winner']['is_enabled']) ){
                $data['auto_winner']['is_enabled'] = 'true' == $data['auto_winner']['is_enabled'] ? 'yes' : 'no';
            }

            if( !isset( $data['is_started'], $data['start_date'] ) || 'yes' != $data['is_started'] ){
                $data['start_date'] = date( 'Y-m-d H:i:s' );
                $data['is_started'] = 'yes';
            }
            
            $data['winner'] = '';
            $result = Wpfnl_Ab_Testing::update_start_settings( $step_id, $data );
            if( $result ){
                Wpfnl_Ab_Testing::update_ab_testing_status( $step_id, 'yes' );
                return rest_ensure_response( $this->get_success_response() );
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }

    /**
     * Update/save start settings
     *
     * @param WP_REST_Request $request
     * @return Array
     * @since 1.6.21
     */
    public function stop_ab_testing( WP_REST_Request $request ){
        if( isset($request['step_id']) ){

            $step_id = $request['step_id'];
            $data = get_post_meta( $step_id, '_wpfnl_ab_testing_start_settings' , true );
            if( isset($data['auto_winner']['is_enabled']) ){
                $data['auto_winner']['is_enabled'] = 'true' == $data['auto_winner']['is_enabled'] ? 'yes' : 'no';
            }
            $data['start_date'] = date( 'Y-m-d H:i:s' );
            $data['is_started'] = '';
            $result = Wpfnl_Ab_Testing::update_start_settings( $step_id, $data );

            if( $result ){
                return rest_ensure_response( $this->get_success_response() );
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }



    


    /**
     * Get start settings 
     * 
     * @param WP_REST_Request $request
     * @return Array 
     * @since 1.6.21
     */
    public function get_start_settings( WP_REST_Request $request ){

        if( isset($request['step_id']) ){
            $step_id = $request['step_id'];
            $result = Wpfnl_Ab_Testing::get_start_settings( $step_id );
            if( $result ){
                $response = $this->get_success_response();
                $response['data']       = $result;
                $start_date = new \DateTime($result['start_date']);
                $now = new \DateTime(Date('Y-m-d'));
                $interval = $start_date->diff($now);
                $response['running_days']=  $interval->d;
                $response['step_type']  = get_post_meta( $step_id, '_step_type', true );
                return rest_ensure_response( $response );
            }
        }

        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * Declear winner
     * 
     * @param WP_REST_Request $request
     * 
     * @return Array 
     * @since  1.6.21
     */
    public function declear_winner( WP_REST_Request $request ){
        if( isset( $request['step_id'] ) ){
            $step_id        = $request['step_id'];
            $variation_id   = isset($request['data']) ? $request['data'] : '';
            $delete_variant   = isset($request['delete_variant']) ? $request['delete_variant'] : '';
            
            $result = Wpfnl_Ab_Testing::set_winner( $step_id,$variation_id );
            if( $result ){
                $response = $this->get_success_response();
                $response['data'] = $variation_id;

                $data = get_post_meta( $step_id, '_wpfnl_ab_testing_start_settings' , true );
                $start_date = '';
                if( $data ){
                    $start_date = isset($data['start_date']) ? $data['start_date'] : '';
                    $data['start_date'] = date( 'Y-m-d H:i:s' );
                    $data['is_started'] = '';
                    Wpfnl_Ab_Testing::update_start_settings( $step_id, $data );
                }

                if( 'yes' == $delete_variant ){
                    $stats_data = isset($request['stats_data']) ? $request['stats_data'] : [];
                    $result = Wpfnl_Ab_Testing::archive_all_variant( $step_id, $variation_id, $start_date, $stats_data );
                    $response['ab_data'] = $result['ab_start_settings'];
                    $response['funnel_data'] = $result['funnel_data'];
                    $response['node_id']     = $result['node_id'];
                    $response['node_data'] = $result['node_data'];
                }
                return rest_ensure_response( $response );
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }
    
    
    /**
     * Archive variant
     * 
     * @param WP_REST_Request $request
     * 
     * @return Array 
     * @since  1.6.21
     */
    public function archive_variant( WP_REST_Request $request ){
        if( isset( $request['step_id'], $request['stats_data'], $request['data'] ) ){
            $step_id        = $request['step_id'];
            $variation_id   = $request['data'];
            $stats_data     = $request['stats_data'];
            if( $step_id && $variation_id &&  $stats_data ){
                $response = $this->get_success_response();
                $data = get_post_meta( $step_id, '_wpfnl_ab_testing_start_settings' , true );
                $start_date = '';
                if( $data ){
                    $start_date = isset($data['start_date']) ? $data['start_date'] : '';
                }
                $result = Wpfnl_Ab_Testing::single_archive( $step_id, $variation_id, $start_date, $stats_data );
                $response['ab_data'] = $result['ab_start_settings'];
                $response['funnel_data'] = $result['funnel_data'];
                $response['node_id']     = $result['node_id'];
                $response['node_data'] = $result['node_data'];
                $response['step_id'] = $result['step_id'];
                $response['is_multiple_variant'] = $result['is_multiple_variant'];
                
                return rest_ensure_response( $response );
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * Get winner
     * 
     * @param WP_REST_Request $request
     * @return Array 
     * @since 1.6.21
     */
    public function get_winner( WP_REST_Request $request ){
        if( isset( $request['step_id'] ) ){
            $step_id        = $request['step_id'];
            $result = Wpfnl_Ab_Testing::get_winner( $step_id );
            if( $result ){
                $response = $this->get_success_response();
                $response['data'] = $result;
                return rest_ensure_response( $response );
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * Delete variant
     * 
     * @param 
     * @since 1.6.21 
     */
    public function delete_variant( WP_REST_Request $request ){
        if( isset( $request['step_id'] ) && isset( $request['variant_id'] ) ){
            $step_id    = $request['step_id'];
            $variant_id = $request['variant_id'];
            $settings   = Wpfnl_Ab_Testing::get_start_settings( $step_id );
            if( isset($settings['variations']) ){
                $key = array_search($variant_id, array_column($settings['variations'], 'id'));
                if( false !== $key ){
                    array_splice($settings['variations'],$key, 1);
                    
                    if( !(count($settings['variations']) > 1) ){
                        $funnel_id = get_post_meta( $step_id, '_funnel_id', true );
                        $funnel_data    = get_post_meta( $funnel_id, 'funnel_data', true );
                        $steps          = $funnel_data['drawflow']['Home']['data'];
                        foreach ( $steps as $key => $step_data ) {
                            $node_data = $step_data['data'];
                            if ( 'conditional' != $node_data['step_type'] && $step_id == $node_data['step_id'] ) {
                                $step_data['class'] = trim(str_replace('has-ab-variation', '', $step_data['class'] ));
                                $funnel_data['drawflow']['Home']['data'][$key] = $step_data;
                            }
                        }
                        update_post_meta($funnel_id, 'funnel_data', $funnel_data);
                        $settings['is_started'] = '';
                        $settings['winner'] = '';
                    }
                    update_post_meta( $step_id, '_wpfnl_ab_testing_start_settings', $settings );
                    $data = Wpfnl_Ab_Testing::maybe_ab_testing( $step_id );
                    
                    $result['is_ab_enabled']  = $data ? $data : '';
                    $result['start_settings'] = Wpfnl_Ab_Testing::get_start_settings( $step_id );
                    $response = $this->get_success_response();
                   
                    $response['abTestingSettingsData']['data'] = $result;
                    return rest_ensure_response( $response );
                }
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * Create varient
     * 
     */
    public function create_varient( WP_REST_Request $request ){
        if( isset( $request['step_id'] ) && isset( $request['funnel_id'] ) && isset( $request['step_name'] ) ){
            $step_id        = $request['step_id'];
            $step_type      = get_post_meta( $step_id, '_step_type', true);
            $funnel_id     = $request['funnel_id'];
            $step_name      = $request['step_name'];

            $funnel = Wpfnl::get_instance()->funnel_store;
            $step = Wpfnl::get_instance()->step_store;

            $varient_id = $step->create_step( $funnel_id, $step_name, $step_type , '' , false );
            $step->set_id($varient_id);

            if ($varient_id && ! is_wp_error($varient_id)) {
                $funnel->set_id($funnel_id);
                $step_edit_link = get_edit_post_link($varient_id);
    
                if( 'elementor' ==  Wpfnl_functions::get_builder_type() ){
                    $step_edit_link = str_replace('/&amp;/g','&',$step_edit_link);
                    $step_edit_link = str_replace('edit','elementor',$step_edit_link);
                }

                $step_view_link = get_post_permalink($varient_id);


                // add has-variation class to that step node of drawflow data
                $funnel_data    = get_post_meta( $funnel_id, 'funnel_data', true );
                $steps          = $funnel_data['drawflow']['Home']['data'];
                foreach ( $steps as $key => $step_data ) {
                    $node_data = $step_data['data'];
                    if ( 'conditional' != $node_data['step_type'] && $step_id == $node_data['step_id'] ) {
                        $step_data['class'] = $step_data['class'].' '.'has-ab-variation';
                        $funnel_data['drawflow']['Home']['data'][$key] = $step_data;
                    }
                }
                update_post_meta($funnel_id, 'funnel_data', $funnel_data);


                Wpfnl_Ab_Testing::update_variations( $step_id, $varient_id );
                update_post_meta( $varient_id, '_parent_step_id', $step_id );
                Wpfnl_Ab_Testing::update_ab_testing_status( $step_id, 'yes');
                return [
                    'success'          		=> true,
                    'step_id'          		=> $step_id,
                    'step_edit_link'   		=> $step_edit_link,
                    'step_view_link'   		=> rtrim( $step_view_link, '/' ),
                    'step_title'       		=> $step->get_title(),
                    'conversion'       		=> 0,
                    'visit'       			=> 0,
                    'shouldShowAnalytics' 	=> false,
                    'abTestingSettingsData' => Wpfnl_Ab_Testing::get_formatted_settings( $step_id )
                ];
            } else {
                return [
                    'success' => false,
                    'message' => $step_id->get_error_message(),
                ];
            }

        }
        return rest_ensure_response( $this->get_error_response() );
    }

    /**
     * Delete archive variant permanently
     * 
     * @param \WP_REST_Request $request
     * 
     * @return Array
     * @since  1.7.5
     */
    public function delete_archive( \WP_REST_Request $request ){
        if( isset( $request['parent_step_id'], $request['step_id'] ) ){
            $variant_id = $request['step_id'];
            $step_id    = $request['parent_step_id'];
            $result = Wpfnl_Ab_Testing::delete_archive( $step_id, $variant_id );
            $ab_testing_data = Wpfnl_Ab_Testing::get_start_settings( $step_id );
            $response = $this->get_success_response();
            $response['ab_data'] = $ab_testing_data;
            return rest_ensure_response( $response );
        }
        return rest_ensure_response( $this->get_error_response() );
    }

    /**
     * Duplicate AB testing varient
     * @param \WP_REST_Request $request
     */
    public function duplicate_varient( \WP_REST_Request $request ){
        if( isset( $request['parent_step_id'] ) && isset( $request['step_id'] )  && isset( $request['funnel_id'] ) && isset( $request['step_name'] ) ){
          
            $step_id        = $request['step_id'];
            $step_type      = get_post_meta( $step_id, '_step_type', true);
            $funnel_id      = $request['funnel_id'];
            $step_name      = $request['step_name'];
            
            $title          = get_the_title($step_id);
            $page_template  = get_post_meta($step_id, '_wp_page_template', true);
            $post_content   = get_post_field('post_content', $step_id);

            $funnel = Wpfnl::get_instance()->funnel_store;
            $step = Wpfnl::get_instance()->step_store;

            $varient_id = $step->create_step($funnel_id, $step_name, $step_type, $post_content, false);
            $step->set_id($varient_id);

            if ($varient_id && ! is_wp_error($varient_id)) {

                $builder = Wpfnl_functions::get_builder_type();
                delete_post_meta($varient_id, '_wp_page_template');
                $step->update_meta($varient_id, '_wp_page_template', $page_template);
                Wpfnl_Ab_Testing::duplicate_ab_testing_meta($step_id, $varient_id, array('_funnel_id', '_wpf_step_title', '_wpf_step_slug'));
                do_action('wpfunnels_after_step_import', $varient_id, $builder);
                $funnel->set_id($funnel_id);

                $step_edit_link = get_edit_post_link($varient_id);

                if( 'elementor' ==  Wpfnl_functions::get_builder_type() ){
                    $step_edit_link = str_replace('/&amp;/g','&',$step_edit_link);
                    $step_edit_link = str_replace('edit','elementor',$step_edit_link);
                }
                
                
                $step_view_link = get_post_permalink($varient_id);

                Wpfnl_Ab_Testing::update_variations( $request['parent_step_id'], $varient_id );
                update_post_meta( $varient_id, '_parent_step_id', $request['parent_step_id'] );
                Wpfnl_Ab_Testing::update_ab_testing_status( $request['parent_step_id'], 'yes');

                // add has-variation class to that step node of drawflow data
                $funnel_data    = get_post_meta( $funnel_id, 'funnel_data', true );
                $steps          = $funnel_data['drawflow']['Home']['data'];
                foreach ( $steps as $key => $step_data ) {
                    $node_data = $step_data['data'];
                    if ( 'conditional' != $node_data['step_type'] && $step_id == $node_data['step_id'] ) {
                        $step_data['class'] = $step_data['class'].' '.'has-ab-variation';
                        $funnel_data['drawflow']['Home']['data'][$key] = $step_data;
                    }
                }

                $ab_testing_data = Wpfnl_Ab_Testing::get_formatted_settings( $request['parent_step_id'] );
                if( isset($ab_testing_data['data']['start_settings']['variations']) && is_array($ab_testing_data['data']['start_settings']['variations']) ){
                    foreach( $ab_testing_data['data']['start_settings']['variations'] as $key => $variation ){
                        if( isset($variation['id'], $variation['step_type']) && ( 'checkout' ==  $variation['step_type'] || 'upsell' ==  $variation['step_type'] || 'downsell' ==  $variation['step_type'])){
                            $step_products = get_post_meta( $variation['id'], '_wpfnl_'.$variation['step_type'].'_products', true );
                            if( is_array($step_products) && count($step_products) ){
                                $ab_testing_data['data']['start_settings']['variations'][$key]['is_product'] = 'yes'; 
                            }
    
                            if( 'checkout' ==  $variation['step_type'] ){
                                $ob_products = get_post_meta( $variation['id'], 'order-bump-settings', true );
                                if( is_array($ob_products) && count($ob_products) ){
                                    $ab_testing_data['data']['start_settings']['variations'][$key]['is_ob'] = 'yes'; 
                                }
                            }
                        }
                    }
                }

                update_post_meta($funnel_id, 'funnel_data', $funnel_data);
                return [
                    'success'          		=> true,
                    'step_id'          		=> $step_id,
                    'step_edit_link'   		=> $step_edit_link,
                    'step_view_link'   		=> rtrim( $step_view_link, '/' ),
                    'step_title'       		=> $step->get_title(),
                    'conversion'       		=> 0,
                    'visit'       			=> 0,
                    'shouldShowAnalytics' 	=> false,
                    'abTestingSettingsData' => $ab_testing_data
                ];
            } else {
                return [
                    'success' => false,
                    'message' => $step_id->get_error_message(),
                ];
            }

        }

        return rest_ensure_response( $this->get_error_response() );
    }
    
    
    /**
     * Restore AB testing archive varient
     * 
     * @param \WP_REST_Request $request
     */
    public function restore_varient( \WP_REST_Request $request ){
        if( isset( $request['parent_step_id'], $request['archive_variation'] ) ){
            $step_id          = $request['step_id'];
            $funnel_id        = Wpfnl_functions::get_funnel_id_from_step( $step_id );
            $archive_variant = $request['archive_variation']; 
            if( $step_id && $funnel_id && $archive_variant ){
                $result = Wpfnl_Ab_Testing::restore_archive_variant( $request['parent_step_id'], $archive_variant['id'], false );
                if( $result ){
                    $funnel_data    = get_post_meta( $funnel_id, 'funnel_data', true );
                    $steps          = $funnel_data['drawflow']['Home']['data'];
                    foreach ( $steps as $key => $step_data ) {
                        $node_data = $step_data['data'];
                        if ( 'conditional' != $node_data['step_type'] && $step_id == $node_data['step_id'] ) {
                            $step_data['class'] = $step_data['class'].' '.'has-ab-variation';
                            $funnel_data['drawflow']['Home']['data'][$key] = $step_data;
                        }
                    }

                    $ab_testing_data = Wpfnl_Ab_Testing::get_formatted_settings( $request['parent_step_id'] );
                    if( isset($ab_testing_data['data']['start_settings']['variations']) && is_array($ab_testing_data['data']['start_settings']['variations']) ){
                        foreach( $ab_testing_data['data']['start_settings']['variations'] as $key => $variation ){
                            if( isset($variation['id'], $variation['step_type']) && ( 'checkout' ==  $variation['step_type'] || 'upsell' ==  $variation['step_type'] || 'downsell' ==  $variation['step_type'])){
                                $step_products = get_post_meta( $variation['id'], '_wpfnl_'.$variation['step_type'].'_products', true );
                                if( is_array($step_products) && count($step_products) ){
                                    $ab_testing_data['data']['start_settings']['variations'][$key]['is_product'] = 'yes'; 
                                }

                                if( 'checkout' ==  $variation['step_type'] ){
                                    $ob_products = get_post_meta( $variation['id'], 'order-bump-settings', true );
                                    if( is_array($ob_products) && count($ob_products) ){
                                        $ab_testing_data['data']['start_settings']['variations'][$key]['is_ob'] = 'yes'; 
                                    }
                                }
                            }
                        }
                    }
                    update_post_meta($funnel_id, 'funnel_data', $funnel_data);
                    return [
                        'success'          		=> true,
                        'abTestingSettingsData' => $ab_testing_data
                    ];
                }
            }
        }
        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * return error reponse
     * @return Array
     * 
     */
    private function get_error_response(){
        return [
            'success' => false,
            'data'    => '',
        ];
    }


    /**
     * return error reponse
     * @return Array
     * 
     */
    private function get_success_response(){

        return [
            'success' => true,
            'data'    => '',
        ];
    }


    /**
     * Prepare a single setting object for response.
     *
     * @param object $item Setting object.
     * @param WP_REST_Request $request Request object.
     * @return \WP_REST_Response $response Response data.
     * @since  1.0.0
     */
    public function prepare_item_for_response($item, $request)
    {
        $data = $this->add_additional_fields_to_object($item, $request);
        return rest_ensure_response($data);
    }


    /**
     * @desc Get statistics data for A/B testing.
     *
     * @param WP_REST_Request $data
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_stats( \WP_REST_Request $data ) {

        if( isset($data['funnel_id'] ) && isset($data['step_id'] ) ){
            $funnel_id = $data->get_param( 'funnel_id' );
            $step_id   = $data->get_param( 'step_id' );
            $step_type   = get_post_meta( $data->get_param( 'step_id' ), '_step_type', true );
            $stats = Wpfnl_Ab_Testing::get_stats( $funnel_id, $step_id );
            $response = $this->get_success_response();
            $response['data'] = $stats;
            $response['step_type'] = $step_type;
            return rest_ensure_response( $response );
        }
        return rest_ensure_response( $this->get_error_response() );
        
    }


    /**
     * @desc Reset analytics stats API callback function
     *
     * @param WP_REST_Request $data
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function reset_stats( \WP_REST_Request $data ) {
        $step_id   = $data->get_param( 'step_id' );
        if( Wpfnl_Ab_Testing::reset_stats( $step_id ) ) {
            return rest_ensure_response( $this->get_success_response() );
        }
        return rest_ensure_response( $this->get_error_response() );
    }


    /**
     * Reset settings API callback function
     *
     * @param WP_REST_Request $data
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function reset_settings( \WP_REST_Request $data ) {
        $step_id   = $data->get_param( 'step_id' );
        if( Wpfnl_Ab_Testing::reset_settings( $step_id ) ) {
            return rest_ensure_response( $this->get_success_response() );
        }
        return rest_ensure_response( $this->get_error_response() );
    }
}
