<?php
namespace WPFunnels\Rest\Controllers;

use WP_REST_Server;
use Wpfnl_Analytics_Factory;
use WPFunnels\Rest\AnalyticsController\TimeInterval;
use WPFunnels\Rest\Controllers\Wpfnl_REST_Controller;
use WPFunnels\Wpfnl_functions;
use WPFunnels\LmsController\LmsController;
use function cli\err;
use WPFunnelsPro\AbTesting\Wpfnl_Ab_Testing;
use WPFunnelsPro\Compatibility\WcHpos;

class AnalyticsController extends Wpfnl_REST_Controller {

    /**
     * Endpoint namespace.
     *
     * @var string
     */
    protected $namespace = 'wpfunnels/v1';

    /**
     * Route base.
     *
     * @var string
     */
    protected $rest_base = 'analytics';


    /**
     * Order by property, used in the cmp function.
     *
     * @var string
     */
    private $order_by = '';
    /**
     * Order property, used in the cmp function.
     *
     * @var string
     */
    private $order = '';


    /**
     * Makes sure the current user has access to READ the settings APIs.
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_Error|boolean
     * @since  3.0.0
     */
    public function get_items_permissions_check($request)
    {
        if ( !Wpfnl_functions::wpfnl_rest_check_manager_permissions('settings') ) {
            return new \WP_Error('wpfunnels_rest_cannot_get', __('Sorry, you cannot list resources.', 'wpfnl'), array('status' => rest_authorization_required_code()));
        }
        return true;
    }


    /**
     * register rest routes
     *
     * @since 1.0.0
     */
    public function register_routes() {
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/get_funnel_type'. '/(?P<funnel_id>\d+)', array(
                'args' => array(
                ),
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'get_funnel_type' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );

        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/(?P<funnel_id>\d+)', array(
                'args' => array(
                ),
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'get_analytics_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );

        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/reset_data/(?P<funnel_id>\d+)', array(
                'args' => array(
                ),
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'reset_analytics_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );
    }


    /**
     * get funnel type
     *
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_funnel_type(\WP_REST_Request $request) {
        $funnel_id  = $request['funnel_id'];
        $steps      = Wpfnl_functions::get_steps($funnel_id);
        $type       = 'lead_funnel';
        if( $steps ) {
            foreach ( $steps as $step ) {
                if( 'checkout' === $step['step_type'] ) {
                    $type = 'sales_funnel';
                    break;
                }
            }
        }
        $response           = array(
            'success'       => true,
            'funnel_type'   => $type
        );
        return rest_ensure_response( $response );
    }


    public function get_analytics_data( \WP_REST_Request $request ) {
        $start_date         = $request->get_param('start_date');
        $end_date           = $request->get_param('end_date');
        $funnel_id          = $request['funnel_id'];
        
        $response           = array(
            'intervals' => '',
            'earnings'  => '',
            'chart'     => '',
            'all_steps' => '',
            'success'   => false,
        );

        if( $funnel_id ){
            $funnel_type = get_post_meta( $funnel_id, '_wpfnl_funnel_type', true);
            $funnel_type = !$funnel_type ? 'wc' : $funnel_type;
            $param_type = wpfnl_pro_analytics_get_param_type($funnel_type);
            if( $param_type ){
                $funnel_orders = $param_type->get_orders_by_funnel( $funnel_id, $start_date, $end_date );
                if( Wpfnl_functions::is_wc_active() && 'wc' == $funnel_type && empty($funnel_orders) ){
                    $wchpos_instance = new WcHpos();
                    if( $wchpos_instance->maybe_hpos_enable() ){
                        $funnel_orders = $wchpos_instance->get_hpos_orders_for_a_funnel( $funnel_id, $start_date, $end_date );
                    }
                }
                $earnings_data      = $param_type->get_earnings( $funnel_id, $funnel_orders, $start_date, $end_date ,'all_earning' );
                $visits             = $this->get_visits( $funnel_id, $start_date, $end_date,$funnel_orders );
                $interval           = $request->get_param('interval');
                $all_steps          = Wpfnl_functions::get_steps($funnel_id);

                global $wpdb;
                $default = array(
                    'start_date'    => TimeInterval::default_after(),
                    'end_date'      => TimeInterval::default_before(),
                    'interval'      => $interval,
                );

                
                $query_args = array(
                    'start_date'    => $request->get_param('start_date').'T00:00:00',
                    'end_date'      => $request->get_param('end_date').'T23:59:59',
                    'interval'      => $interval,
                );

                $query_args = wp_parse_args( $query_args, $default );
                $this->normalize_timezones($query_args, $default );
                $db_intervals               = $param_type->get_intervals( $funnel_id, $request->get_param('start_date'), $request->get_param('end_date'), true, $request->get_param('interval') );
                $expected_interval_count    = TimeInterval::intervals_between( $query_args['start_date'], $query_args['end_date'], $query_args['interval'] );
                $db_interval_count          = count( $db_intervals );
                $intervals = $param_type->get_intervals( $funnel_id, $request->get_param('start_date'), $request->get_param('end_date'), false, $request->get_param('interval') );
                
                $data = (object) array(
                    'earnings'  => $earnings_data,
                    'intervals' => $intervals,
                    'all_steps'  => $visits,
                );
                $earn = $data->intervals;
                if ( TimeInterval::intervals_missing( $expected_interval_count, $db_interval_count ) ) {
                    $param_type->fill_in_missing_intervals( $db_intervals, $query_args['start_date'], $query_args['end_date'], $query_args['interval'], $data, $funnel_id );
                    $this->sort_intervals( $data, 'date', 'asc' );
                }
                
                $char_data = $this->get_chart_data($data->intervals);
                
                $visitIndex = 0;
                $i = 0;
                $steps = array();
                foreach($all_steps as $key=>$step){
                    if($step['step_type'] != 'conditional'){
                        $steps[$i]['id']        = $step['id'];
                        $steps[$i]['step_type'] = $step['step_type'];
                        $steps[$i]['name']      = $step['name'];
                        $i++;
                    }
                }
                
                if(count($steps) == count($visits)){
                    foreach($steps as $step){
                        $steps[$visitIndex]['total_visits'] = $visits[$visitIndex]['total_visits'];
                        $steps[$visitIndex]['returning_visits'] = isset($visits[$visitIndex]['returning_visits']) ? $visits[$visitIndex]['returning_visits']: 0;
                        $steps[$visitIndex]['unique_visits'] = isset($visits[$visitIndex]['unique_visits']) ? $visits[$visitIndex]['unique_visits']: 0;
                        $steps[$visitIndex]['conversions'] = isset($visits[$visitIndex]['conversions']) ? $visits[$visitIndex]['conversions']: 0;
                        $steps[$visitIndex]['new_conversions'] = isset($visits[$visitIndex]['new_conversions']) ? $visits[$visitIndex]['new_conversions']: 0;
                        $steps[$visitIndex]['conversions_rate'] = isset($visits[$visitIndex]['conversions_rate']) ? $visits[$visitIndex]['conversions_rate'] : 0;
                        $steps[$visitIndex]['new_conversions_rate'] = isset($visits[$visitIndex]['new_conversions_rate']) ? $visits[$visitIndex]['new_conversions_rate']: 0;
                        $visitIndex++;
                    }
                }


                foreach($data->intervals as $key=>$interval){
                    
                    if( isset($earn[0]['time_interval']) && isset($interval['date_start']) && isset($interval['date_end']) && $interval['date_start'] <= $earn[0]['time_interval'] && $interval['date_end'] >= $earn[0]['time_interval']){
                        $data->intervals[$key]['order'] = isset($earn[0]['order']) ? $earn[0]['order'] :0;
                        $data->intervals[$key]['gross_sale'] = isset($earn[0]['gross_sale'])?$earn[0]['gross_sale']:0 ;
                        $data->intervals[$key]['order_bump'] = isset($earn[0]['order_bump'])? $earn[0]['order_bump']:0;
                        $data->intervals[$key]['avg_order_value'] = isset($earn[0]['avg_order_value'])?$earn[0]['avg_order_value']:0 ;
                        $data->intervals[$key]['revenue'] = isset($earn[0]['revenue'])?$earn[0]['revenue']:0 ;

                    }
                }
                
                $response           = array(
                    'intervals' => $data->intervals,
                    'earnings'  => $data->earnings,
                    'chart'     => $char_data,
                    'all_steps' => $steps,
                    'success'   => true,
                );
            }
            
        }
        return rest_ensure_response( $response );
    }

    /**
     * get user visit data
     *
     * @param $funnel_id
     */
    private function get_visits( $funnel_id, $start_date, $end_date, $funnel_orders ) {
       
        // $start_date = date("d-m-Y H:ii:s", strtotime($start_date));
        // $end_date = date("d-m-Y H:i:s", strtotime($end_date));
        
        global $wpdb;
        $analytics_db       = $wpdb->prefix . WPFNL_PRO_ANALYTICS_TABLE;
        $analytics_meta_db  = $wpdb->prefix . WPFNL_PRO_ANALYTICS_META_TABLE;
        // get all steps
        $steps     = Wpfnl_functions::get_steps($funnel_id);
        $all_steps = array();

        foreach ( $steps as $key => $step ) {
            if($step['step_type'] != 'conditional'){
                $all_steps[] = $step['id'];
                $variations = Wpfnl_Ab_Testing::get_all_variations( $step['id'] );
                if( is_array($variations) && count($variations) > 1 ){
                    foreach( $variations as $variation ){
                        if( isset($variation['id']) && !in_array( $variation['id'], $all_steps ) ){
                            $all_steps[] = $variation['id'];
                        }
                    }
                }
                
            }
        }

        $step_ids       = implode( ', ', $all_steps );
         
       
        $date       = date('Y-m-d');
        $analytics_columns  = array(
            'step_id'       => "wpft1.step_id",
            'total_visits'  => "COUNT( DISTINCT( wpft1.id ) ) AS total_visits",
            'unique_visits' => "COUNT( DISTINCT( CASE WHEN wpft1.visitor_type = 'new' THEN wpft1.id ELSE NULL END ) ) AS unique_visits",
            'conversion'    => "COUNT( CASE WHEN wpft2.meta_key = 'conversion' AND wpft2.meta_value = 'yes' AND wpft1.visitor_type = 'returning' THEN wpft1.step_id ELSE NULL END ) AS conversions ",
            'new_conversion'=> "COUNT( CASE WHEN wpft2.meta_key = 'conversion' AND wpft2.meta_value = 'yes' AND wpft1.visitor_type = 'new' THEN wpft1.step_id ELSE NULL END ) AS new_conversions "
        );
        $end_date = date('Y-m-d', strtotime('+1 day', strtotime($end_date)));
        $where = '';
        $where .= " AND wpft1.date_created >= %s ";
        $where .= " AND wpft1.date_created < %s ";
        $query = $wpdb->prepare(
            "SELECT {$analytics_columns['step_id']},
            {$analytics_columns['total_visits']},
            {$analytics_columns['unique_visits']},
            {$analytics_columns['conversion']},
            {$analytics_columns['new_conversion']}
            FROM $analytics_db as wpft1 INNER JOIN $analytics_meta_db as wpft2 ON wpft1.id = wpft2.analytics_id
            WHERE ( wpft1.step_id IN ( $step_ids ) )
            {$where}
            GROUP BY wpft1.step_id
            ORDER BY NULL
            ",$start_date,$end_date);

        
        $visits_data        = $wpdb->get_results( $query );
   
       
        $total_visit_data   = array();
        if($steps) {
            
            foreach ($steps as $step) {
                if($step['step_type'] != 'conditional'){
                    if($visits_data) {
                        
                        $variations = Wpfnl_Ab_Testing::get_all_variations( $step['id'] );
                        if( is_array($variations) && count($variations) > 1 ){
                            $key = array_search( $step['id'], array_column($visits_data, 'step_id'));
                            $variation_total_visit = 0;
                            $variation_returning_visits = 0;
                            $variation_unique_visits = 0;
                            $variation_conversions = 0;
                            $variation_new_conversions = 0;
                            $conversion_rate = 0;
                            $new_conversion_rate = 0;
                            foreach( $variations as $variant_key => $variation ){
                                $variant_key = array_search( $variation['id'], array_column($visits_data, 'step_id'));
                                $isPermittedData = false;
                                
                                if($variant_key!==false){
                    
                                    $variation_total_visit += $visits_data[$variant_key]->total_visits;
                                    $variation_returning_visits += ($visits_data[$variant_key]->total_visits-$visits_data[$variant_key]->unique_visits);
                                    $variation_unique_visits += $visits_data[$variant_key]->unique_visits;
                                    $variation_conversions += $visits_data[$variant_key]->conversions;
                                    $variation_new_conversions += $visits_data[$variant_key]->new_conversions;

                                    if ( $variation_conversions > 0 ) {
                                        if($variation_returning_visits > 0){
                                            $conversion_rate        = $variation_conversions / intval($variation_returning_visits) * 100;
                                        }else{
                                            $conversion_rate        = 0;
                                        }
            
                                        if($variation_unique_visits > 0){
                                            $new_conversion_rate    = $variation_new_conversions / intval($variation_unique_visits) * 100;
                                        }else{
                                            $new_conversion_rate    = 0;
                                        }
                            
                                    }
                                }
                            }
                            $total_visit_data[] = array(
                                'step_id'               => $visits_data[$key]->step_id,
                                'step_name'             => $step['name'],
                                'total_visits'          => $variation_total_visit,
                                'returning_visits'      => $variation_returning_visits,
                                'unique_visits'         => $variation_unique_visits,
                                'conversions'           => $variation_conversions,
                                'new_conversions'       => $variation_new_conversions,
                                'conversions_rate'      => $conversion_rate,
                                'new_conversions_rate'  => $new_conversion_rate,
                                
                            );
                        }else{
                            $key = array_search( $step['id'], array_column($visits_data, 'step_id'));
                            $isPermittedData = false;
                            if($key!==false){
                                if ( $visits_data[$key]->total_visits > 0 ) {
                                    if($visits_data[$key]->total_visits-$visits_data[$key]->unique_visits > 0){
                                        $conversion_rate        = $visits_data[$key]->conversions / intval($visits_data[$key]->total_visits-$visits_data[$key]->unique_visits) * 100;
                                    }else{
                                        $conversion_rate        = 0;
                                    }
        
                                    if($visits_data[$key]->unique_visits > 0){
                                        $new_conversion_rate    = $visits_data[$key]->new_conversions / intval($visits_data[$key]->unique_visits) * 100;
                                    }else{
                                        $new_conversion_rate    = 0;
                                    }
                        
                                }
                                $total_visit_data[] = array(
                                    'step_id'               => $visits_data[$key]->step_id,
                                    'step_name'             => $step['name'],
                                    'total_visits'          => $visits_data[$key]->total_visits,
                                    'returning_visits'      => $visits_data[$key]->total_visits-$visits_data[$key]->unique_visits,
                                    'unique_visits'         => $visits_data[$key]->unique_visits,
                                    'conversions'           => $visits_data[$key]->conversions,
                                    'new_conversions'       => $visits_data[$key]->new_conversions,
                                    'conversions_rate'      => $conversion_rate,
                                    'new_conversions_rate'  => $new_conversion_rate,
                                    
                                );
                            }
                            else{
                                $total_visit_data[] = array(
                                    'step_id'               => $step['id'],
                                    'step_name'             => $step['name'],
                                    'total_visits'          => 0,
                                    'returning_visits'      => 0,
                                    'unique_visits'         => 0,
                                    'conversions'           => 0,
                                    'new_conversions'       => 0,
                                    'conversions_rate'      => 0,
                                    'new_conversions_rate'  => 0,
                                    
                                );
                            }
                        }

                        
                        
                    } else {
                        $total_visit_data[] = array(
                            'step_id'               => $step['id'],
                            'step_name'             => $step['name'],
                            'total_visits'          => 0,
                            'returning_visits'      => 0,
                            'unique_visits'         => 0,
                            'conversions'           => 0,
                            'new_conversions'       => 0,
                            'conversions_rate'      => 0,
                            'new_conversions_rate'  => 0,
                           
                        );
                    }
                }
            }
        }
        return $total_visit_data;
    }


    /**
     * Converts input datetime parameters to local timezone. If there are no inputs from the user in query_args,
     * uses default from $defaults.
     *
     * @param $query_args
     * @param $defaults
     * @throws \Exception
     *
     * @since 1.0.0
     */
    protected function normalize_timezones( &$query_args, $defaults ) {
        $local_tz = new \DateTimeZone( Wpfnl_functions::wpfnl_timezone_string() );
        foreach ( array( 'end_date', 'start_date' ) as $query_arg_key ) {
            if ( isset( $query_args[ $query_arg_key ] ) && is_string( $query_args[ $query_arg_key ] ) ) {
                // Assume that unspecified timezone is a local timezone.
                
                $datetime = new \DateTime( $query_args[ $query_arg_key ], $local_tz );
                // In case timezone was forced by using +HH:MM, convert to local timezone.
                $datetime->setTimezone( $local_tz );
                $query_args[ $query_arg_key ] = $datetime;
            } elseif ( isset( $query_args[ $query_arg_key ] ) && is_a( $query_args[ $query_arg_key ], 'DateTime' ) ) {
                // In case timezone is in other timezone, convert to local timezone.
                $query_args[ $query_arg_key ]->setTimezone( $local_tz );
            } else {
                $query_args[ $query_arg_key ] = isset( $defaults[ $query_arg_key ] ) ? $defaults[ $query_arg_key ] : null;
            }
        }
    }


    /**
     * Sorts intervals according to user's request.
     *
     * They are pre-sorted in SQL, but after adding gaps, they need to be sorted including the added ones.
     *
     * @param \stdClass $data      Data object, must contain an array under $data->intervals.
     * @param string   $sort_by   Ordering property.
     * @param string   $direction DESC/ASC.
     */
    protected function sort_intervals( &$data, $sort_by, $direction ) {
        $this->sort_array( $data->intervals, $sort_by, $direction );
    }


    /**
     * Sorts array of arrays based on subarray key $sort_by.
     *
     * @param array  $arr       Array to sort.
     * @param string $sort_by   Ordering property.
     * @param string $direction DESC/ASC.
     */
    protected function sort_array( &$arr, $sort_by, $direction ) {
        $this->order_by = $this->normalize_order_by( $sort_by );
        $this->order    = $direction;
        usort( $arr, array( $this, 'interval_cmp' ) );
    }

    /**
     * Normalizes order_by clause to match to SQL query.
     *
     * @param string $order_by Order by option requested by user.
     * @return string
     */
    protected function normalize_order_by( $order_by ) {
        if ( 'date' === $order_by ) {
            return 'time_interval';
        }

        return $order_by;
    }


    /**
     * Compares two report data objects by pre-defined object property and ASC/DESC ordering.
     *
     * @param stdClass $a Object a.
     * @param stdClass $b Object b.
     * @return string
     */
    private function interval_cmp( $a, $b ) {
        if ( '' === $this->order_by || '' === $this->order ) {
            return 0;
            // @todo Should return WP_Error here perhaps?
        }
        if ( $a[ $this->order_by ] === $b[ $this->order_by ] ) {
            // As relative order is undefined in case of equality in usort, second-level sorting by date needs to be enforced
            // so that paging is stable.
            if ( $a['time_interval'] === $b['time_interval'] ) {
                return 0; // This should never happen.
            } elseif ( $a['time_interval'] > $b['time_interval'] ) {
                return 1;
            } elseif ( $a['time_interval'] < $b['time_interval'] ) {
                return -1;
            }
        } elseif ( $a[ $this->order_by ] > $b[ $this->order_by ] ) {
            return strtolower( $this->order ) === 'desc' ? -1 : 1;
        } elseif ( $a[ $this->order_by ] < $b[ $this->order_by ] ) {
            return strtolower( $this->order ) === 'desc' ? 1 : -1;
        }
    }


    public function get_chart_data($intervals) {
        $chart_data = array(
            'labels'            => [],
            'conversion'        => [],
            'conversion_rate'   => [],
            'revenue'           => [],
        );
        if($intervals) {
            foreach ( $intervals as $interval ) {
                $chart_data['labels'][] = $interval;
            }
        }
    }


    /**
     * Reset analytics data
     * 
     * @param $request
     */
    public function reset_analytics_data(\WP_REST_Request $request){
        

        if( isset($request['funnel_id']) ){
            $funnel_id = $request['funnel_id'];
            $funnel_type = get_post_meta( $funnel_id, '_wpfnl_funnel_type', true);
            $param_type = wpfnl_pro_analytics_get_param_type($funnel_type);
            if( $param_type ){
                $response = $param_type->reset_analytics_data($request);
            }
        } 
        return rest_ensure_response( $response );
    }
}


if( !function_exists('wpfnl_pro_analytics_get_param_type') ) {
    function wpfnl_pro_analytics_get_param_type( $param ) {
        $param_type = Wpfnl_Analytics_Factory::build($param);
        return $param_type;
    }
}
