<?php


namespace WPFunnels\Rest\Controllers;

use Mint\MRM\DataBase\Tables\AutomationMetaSchema;
use WP_REST_Server;
use WPFunnels\Wpfnl_functions;
use WPFunnelsPro\Wpfnl_Pro_functions;


class MintController extends Wpfnl_REST_Controller {

    /**
     * Endpoint namespace.
     *
     * @var string
     */
    protected $namespace = 'wpfunnels/v1';

    /**
     * Route base.
     *
     * @var string
     */
    protected $rest_base = 'mint/automation';


    /**
     * Order by property, used in the cmp function.
     *
     * @var string
     */
    private $order_by = '';


    /**
     * Order property, used in the cmp function.
     *
     * @var string
     */
    private $order = '';


    /**
     * Makes sure the current user has access to READ the settings APIs.
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_Error|boolean
     * @since  3.0.0
     */
    public function get_items_permissions_check($request)
    {
        return true;
        if ( !Wpfnl_functions::wpfnl_rest_check_manager_permissions('settings') ) {
            return new \WP_Error('wpfunnels_rest_cannot_get', __('Sorry, you cannot list resources.', 'wpfnl'), array('status' => rest_authorization_required_code()));
        }
        return true;
    }


    /**
     * register rest routes
     *
     * @since 1.0.0
     */
    public function register_routes() {

        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/(?P<funnel_id>\d+)', array(
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'get_automations' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );
        
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/get_automation/(?P<funnel_id>\d+)', array(
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'get_automation_by_id' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );

        
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/save_data/(?P<step_id>\d+)', array(
                array(
                    'methods'               => 'POST',
                    'callback'              => array( $this, 'save_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args' => array(
                        'funnel_id' => array(
                            'description'   => __('Funnel ID.', 'wpfnl-pro'),
                            'type'          =>   'string',
                        )
                    ),
                )
            )
        );

        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/save_mail_data', array(
                array(
                    'methods'               => 'POST',
                    'callback'              => array( $this, 'save_mail_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args' => array(
                        'funnel_id' => array(
                            'description'   => __('Funnel ID.', 'wpfnl-pro'),
                            'type'          =>   'string',
                        )
                    ),
                )
            )
        );
        
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/get_mail_data/(?P<step_id>\d+)/(?P<index>\d+)', array(
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'get_mail_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );
        
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/get_mail_template_data/(?P<index>\d+)', array(
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'get_mail_template_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );
        
        
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/render_email_builder', array(
                array(
                    'methods'               => WP_REST_Server::READABLE,
                    'callback'              => array( $this, 'render_email_builder' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args'                  => $this->get_endpoint_args_for_item_schema(WP_REST_Server::READABLE),
                )
            )
        );
        
        register_rest_route(
            $this->namespace, '/' . $this->rest_base . '/get_data/(?P<step_id>\d+)/(?P<index>\d+)', array(
                array(
                    'methods'               => 'GET',
                    'callback'              => array( $this, 'get_data' ),
                    'permission_callback'   => array( $this, 'get_items_permissions_check' ),
                    'args' => array(
                        'funnel_id' => array(
                            'description'   => __('Funnel ID.', 'wpfnl-pro'),
                            'type'          =>   'string',
                        )
                    ),
                )
            )
        );

    }

    /**
     * Get Automation data from DB by funnel id
     *
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_automation_data( \WP_REST_Request $request ) {

        $funnel_id              = $request['funnel_id'];
        $is_automation_enabled  = get_post_meta( $funnel_id, 'is_automation_enabled', true );
        $automation_data        = get_post_meta( $funnel_id, 'funnel_automation_data', true );
        if( !$automation_data ) {
            $automation_data = array();
        }
        $response               = array(
            'status'                => 'success',
            'is_automation_enabled' => $is_automation_enabled,
            'automation_data'       => $automation_data,
        );
        return rest_ensure_response( $response );
    }

    
    /**
     * Save automation data to postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function save_automation_data( \WP_REST_Request $request ){

        $funnel_id  = $request['funnel_id'] ? $request['funnel_id'] : '';
        $step_id  = $request['step_id'] ? $request['step_id'] : '';
        $params = Wpfnl_functions::get_default_automation();
        try {
            $class_name = "Mint\\MRM\\Automation\AutomationModel";
            if( class_exists($class_name) ){
                $automation_id = $class_name::get_instance()->create_or_update( $params );
                if ( $automation_id ) {
                    $data = array(
                        'automation_id' => $automation_id,
                    );
                    update_post_meta( $step_id, '_wpfnl_autoamtion_id', $automation_id );
                    $this->update_meta( $automation_id, 'source', 'wpf' );
                    return $this->get_success_response( __( 'Automation has been saved successfully', 'mrm' ), 201, $data );
                }
                return $this->get_error_response( __( 'Failed to save', 'mrm' ), 400 );
            }
            return $this->get_error_response( __( 'Failed to save automation step', 'mrm' ), 400 );
		} catch ( Exception $e ) {
			return $this->get_error_response( __( 'Failed to save automation step', 'mrm' ), 400 );
		}
    }
    
    
    /**
     * Save automation data to postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function save_data( \WP_REST_Request $request ){

        $step_id        = isset($request['step_id']) ? $request['step_id'] : '';
        $settings       = isset($request['settings']) ? $request['settings'] : '';
        $type           = isset($request['type']) ? $request['type'] : '';
        $groups         = isset($request['groups']) ? $request['groups'] : '';
        $group_type     = isset($request['group_type']) ? $request['group_type'] : '';
        $index          = isset($request['index']) ? $request['index'] : '';
        $selected_trigger          = isset($request['selected_trigger']) ? $request['selected_trigger'] : [];
        try {
            $step_settings = get_post_meta( $step_id, '_wpfnl_automation_steps', true);
            if( is_array($step_settings) && isset($step_settings[$index]['key'])){
                
                if( 'addList' === $step_settings[$index]['key'] || 'addTag' === $step_settings[$index]['key'] || 'removeList' === $step_settings[$index]['key'] || 'removeTag' === $step_settings[$index]['key'] ){
                    $settings = $this->group_settings( $group_type, $step_id, $groups );
                }elseif( 'delay' === $step_settings[$index]['key'] ){
                    $settings = $this->delay_settings( $step_id, $settings );
                }elseif( 'sequence' === $step_settings[$index]['key'] ){
                    $settings = $this->sequence_settings( $step_id, $settings );
                }elseif( 'sendMail' ===$step_settings[$index]['key'] ){
                    $settings = $this->mail_settings( $step_id, $settings );
                }
            }
            $step_settings[$index]['settings'] = $settings;
           
            update_post_meta( $step_id, '_wpfnl_automation_steps', $step_settings );
            if( !empty( $selected_trigger['trigger'] ) ){
                update_post_meta( $step_id, '_wpfnl_automation_trigger', $selected_trigger );
            }
            $key = array_search('sendMail', array_column($step_settings, 'key'));
            if( false !== $key && isset($step_settings[$key]['settings']['settings']['message_data']['body'])  ){
               unset( $step_settings[$key]['settings']['settings']['message_data']['body'] );
            }
            return $this->get_success_response( __( 'Automation has been saved successfully', 'mrm' ), 201, $step_settings );
            
		} catch ( \Exception $e ) {
			return $this->get_error_response( __( 'Failed to save automation step', 'mrm' ), 400 );
		}
    }
    
    
    
    /**
     * Save automation data to postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function save_mail_data( \WP_REST_Request $request ){
        $step_id  = $request['step_id'] ? $request['step_id'] : '';
        $index  = $request['index'] ? $request['index'] : 0;
        try {
            if( $step_id ){
                $settings = [];
                $step_settings = get_post_meta( $step_id, '_wpfnl_automation_steps', true );
                if( isset($step_settings[$index]['settings']['settings']) ){
                    $settings = $step_settings[$index]['settings'];
                }
                if( !$settings ){
                    $settings = [];
                }

                $settings['settings']['message_data']['body']       = isset($request['email_body']) ? $request['email_body'] : '';
                $settings['settings']['message_data']['json_body']  = isset($request['json_data']) ? $request['json_data'] : '';
                $step_settings[$index]['settings'] = $settings;
                update_post_meta( $step_id, '_wpfnl_automation_steps', $step_settings );
                update_post_meta( $step_id, '_wpfnl_automation_email_body', $settings['settings']['message_data']['body'] );
            }
            return $this->get_success_response( __( 'Automation has been saved successfully', 'mrm' ), 201, [] );
		} catch ( \Exception $e ) {
			return $this->get_error_response( __( 'Failed to save automation step', 'mrm' ), 400 );
		}
    }


    /**
     * Save automation data to postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_mail_data( \WP_REST_Request $request ){
        $step_id  = $request['step_id'] ? $request['step_id'] : '';
        $index  = $request['index'] ? $request['index'] : 0;
        try {
            $step_settings = get_post_meta( $step_id, '_wpfnl_automation_steps', true );
            if( isset($step_settings[$index]['settings']['settings']) ){
                $settings = $step_settings[$index]['settings'];
                if( $settings ){
                    $response = [
                        // 'email_body'    => isset($settings['settings']['message_data']['body']) ? $settings['settings']['message_data']['body'] : '',
                        'json_data'     => isset($settings['settings']['message_data']['json_body']) ? $settings['settings']['message_data']['json_body'] : [],
                    ];
                    return $this->get_success_response( __( 'Automation has been saved successfully', 'mrm' ), 201, $response );
                }
            }
            return $this->get_error_response( __( 'Failed to get data', 'wpfnl' ), 400 );
		} catch ( \Exception $e ) {
			return $this->get_error_response( __( 'Failed to get data', 'wpfnl' ), 400 );
		}
    }
    
    /**
     * Save automation data to postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_mail_template_data( \WP_REST_Request $request ){
        $index  = $request['index'] ? $request['index'] : 0;
        try {
           
            $templates = \Mint\MRM\Internal\Admin\EmailTemplates\DefaultEmailTemplates::get_default_templates();
            $response = [
                'json_data'     => isset($templates[$index]['json_content']) ? $templates[$index]['json_content'] : [],
            ];
            return $this->get_success_response( __( 'Success', 'WPFNL' ), 201, $response );
		} catch ( \Exception $e ) {
			return $this->get_error_response( __( 'Failed to get data', 'wpfnl' ), 400 );
		}
    }


    /**
     * Save automation data to postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function render_email_builder( \WP_REST_Request $request ){
        $url = admin_url().'admin.php?page=email-builder';
        $current_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
        $response = [
            'url'           => $url,
            'current_url'   => $current_url,
        ];
        return $this->get_success_response( __( 'Automation has been saved successfully', 'mrm' ), 201, $response );
        
    }


    /**
     * prepare offer settings for mint automation
     */
    private function offer_settings( $step_id, $funnel_id, $settings, $type, $next_step_id){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => $type,
            'settings'  => [
                'type'                  => 'selected',
                'selected_funnel_ids'   => [$funnel_id],
                'all_steps'             => 'no',
                'selectedStep'          => $settings['selectedStep'],
                'selectedAction'        => $settings['selectedAction'],
                'selectedType'          => $settings['selectedType']
            ],
            'next_step_id' => $next_step_id
        ];
        return $settings;
    }


    /**
     * Prepare tag/list settings
     */
    private function group_settings( $settings_type, $step_id, $groups ){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => 'action',
        ];
        $settings['settings'][$settings_type.'_settings'][$settings_type.'s'] = $groups;
        return $settings;
    }
    
    
    /**
     * Prepare delay settings
     */
    private function delay_settings( $step_id, $settings ){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => 'action',
            'settings'  => [
                'delay_settings' => $settings
            ],
        ];
        return $settings;
    }
    
    /**
     * Prepare sequence settings
     */
    private function sequence_settings( $step_id, $settings ){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => 'action',
            'settings'  => [
                'sequence_settings' => $settings
            ],
        ];
        return $settings;
    }

    /**
     * Prepare mail settings
     */
    private function mail_settings( $step_id, $settings ){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => 'action',
            'settings'  => [
                'message_data' => $settings
            ],
        ];
        return $settings;
    }
    
    
    /**
     * Prepare optin/cta settings
     */
    private function general_settings( $step_id, $funnel_id, $settings, $next_step_id ){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => 'trigger',
            'settings'  => [
                'type'                  => 'selected',
                'selected_funnel_ids'   => [$funnel_id],
                'all_steps'             => 'no',
                'selectedStep'          => isset($settings['selectedStep']) ? $settings['selectedStep'] : '',
            ],
            'next_step_id' => $next_step_id
        ];
        return $settings;
    }
    
    
    /**
     * Prepare optin/cta settings
     */
    private function ob_settings( $step_id, $funnel_id, $settings, $next_step_id ){
        $step_type = get_post_meta( $step_id,'_step_type', true);
        $settings = [
            'step_id'   => $step_id,
            'key'       => $step_type,
            'type'      => 'trigger',
            'settings'  => [
                'type'                  => 'selected',
                'selected_funnel_ids'   => [$funnel_id],
                'selectedOb'            => $settings['selectedOb'],
            ],
            'next_step_id' => $next_step_id
        ];
        return $settings;
    }


    /**
	 * Get next step of the
	 * funnel
	 *
	 * @param $funnel_id
	 * @param $step_id
	 * @param bool $condition
	 * 
	 * @return Mix
	 *
	 * @since 1.0.0
	 */
	public function get_next_automation_step( $funnel_id, $step_id, $condition = true )
	{
		if( $funnel_id && !$step_id ) {
			return false;
		}
		$parent_step_id =  get_post_meta( $step_id, '_parent_step_id', true );
		if( $parent_step_id ){
			$step_id = $parent_step_id;
		}
		$funnel_data = Wpfnl_functions::get_funnel_data($funnel_id);
		if ( $funnel_data ) {
			$node_id        = Wpfnl_functions::get_node_id( $funnel_id, $step_id );

			$node_data      = $funnel_data['drawflow']['Home']['data'];
			$step_type =  get_post_meta( $step_id, '_step_type', true );
         
            $triggers = Wpfnl_functions::get_mint_triggers();
			foreach ( $node_data as $node_key => $node_value ) {
				if ( $node_value['id'] == $node_id ) {
					
					if( in_array( $step_type, $triggers ) ){
						$next_node_id 	= isset($node_value['outputs']['output_2']['connections'][0]['node']) ? $node_value['outputs']['output_2']['connections'][0]['node'] : '';
					}else{
						if( $condition ) {
							$next_node_id 	= isset($node_value['outputs']['output_1']['connections'][0]['node']) ? $node_value['outputs']['output_1']['connections'][0]['node'] : '';
						} else {
							$next_node_id 	= isset($node_value['outputs']['output_2']['connections'][0]['node']) ? $node_value['outputs']['output_2']['connections'][0]['node'] : '';
						}
					}
					
					if( $next_node_id ){
						$next_step_id 	= Wpfnl_functions::get_step_by_node( $funnel_id, $next_node_id );
						$next_step_type = Wpfnl_functions::get_node_type( $node_data, $next_node_id );
						
						return apply_filters( 'wpfunnels/next_step_data', array(
							'step_id' 	=> $next_step_id,
							'step_type' => $next_step_type,
						)); 
					}else{
						// if there is no thank you page
						Wpfnl_functions::redirect_to_deafult_thankyou();
					}
				}
			}
		}
		return false;
	}


    /**
     * Get automation data from postmeta
     * 
     * @param \WP_REST_Request $request
     * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_data( \WP_REST_Request $request ){
        $step_id  = $request['step_id'] ? $request['step_id'] : '';
        $index  = $request['index'] ? $request['index'] : 0;
        try {
            $step_settings = get_post_meta( $step_id, '_wpfnl_automation_steps', true );
            $settings = [];
            if( isset($step_settings[$index]['settings']['settings']) ){
                $settings = $step_settings[$index]['settings'];
            }
           
            if( isset($settings['settings']['message_data']['body']) ){
                $settings['settings']['message_data']['body'] = '';
            }   
            $selected_trigger = get_post_meta( $step_id, '_wpfnl_automation_trigger', true );
            if( !$selected_trigger ){
                $selected_trigger = [
                    'trigger' => '',
                    'condition' => '',
                ];
            }
            $response = [
                'data'      => $settings,
                'trigger'   => $selected_trigger,
            ];
           
            return $this->get_success_response( __( 'success', 'wpfnl' ), 201, $response );
            return $this->get_error_response( __( 'Failed to get data', 'wpfnl' ), 400 );
		} catch ( \Exception $e ) {
			return $this->get_error_response( __( 'Failed to get data', 'wpfnl' ), 400 );
		}
    }


    /**
	 * Update/insert automation meta
	 * 
	 * @param int $automation_id
	 * @param string $meta_key
	 * @param string $meta_value
	 */
	public function update_meta( $automation_id,$meta_key,$meta_value ){

		global $wpdb;
		$automation_meta_table = $wpdb->prefix . AutomationMetaSchema::$table_name;
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$select_query = $wpdb->prepare( "SELECT * FROM $automation_meta_table WHERE automation_id = %d AND meta_key = %s", array( $automation_id, $meta_key ) );
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
		$results = $wpdb->get_results( $select_query ); // db call ok. ; no-cache ok.
		// phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
       
		if( $results ){
			try{
				$payload = [
					'id' 			=> isset($results[0]['id']) ? $results[0]['id'] : '',
					'meta_key'		=> $meta_key,
					'meta_value'	=> $meta_value,
				];
				$payload['updated_at'] = current_time( 'mysql' );
				$updated = $wpdb->update(
					$automation_meta_table,
					$payload,
					array( 'ID' => $payload['id'] )
				); // db call ok. ; no-cache ok.
	
				if( $updated ){
					return true;
				}else{
					return false;
				}
			}catch( \Exception $e ){
				return false;
			}
		}else{
			try{
				$wpdb->insert(
				$automation_meta_table,
					array(
						'automation_id'         => $automation_id,
						'meta_key'       		=> $meta_key,
						'meta_value'		 	=> $meta_value,
						'created_at'   			=> current_time( 'mysql' ),
						'updated_at'   			=> current_time( 'mysql' ),
					)
				); // db call ok.
				return $wpdb->insert_id;
			}catch( \Exception $e ){
				return false;
			}
		}
	}



    /**
	 * Prepare success response for REST API
	 *
	 * @param string $message Response success message.
	 * @param int    $code Response success code.
	 * @param mixed  $data Response data on success.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function get_success_response( $message = '', $code = 0, $data = null ) {
		$response = array(
			'code'    => $code,
			'message' => $message,
			'success' => true,
			'data'    => $data,
		);

		return rest_ensure_response( $response );
	}


	/**
	 * Prepare error response for REST API
	 *
	 * @param string $message Response error message.
	 * @param int    $code Response error code.
	 * @param mixed  $wp_error Response data on error.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function get_error_response( $message = '', $code = 0, $wp_error = null ) {
		return array(
            'success'       => false,
        );
	}

}