<?php

namespace WPFunnelsPro\Frontend\Modules\Upsell;

use WPFunnels\Frontend\Module\Wpfnl_Frontend_Module;
use WPFunnels\Wpfnl_functions;
use WPFunnelsPro\Wpfnl_Pro_functions;
use WPFunnels\lms\helper\Wpfnl_lms_learndash_functions;
use WPFunnelsPro\Frontend\Modules\Gateways\Payment_Gateways_Factory;
use WPFunnelsPro\Wpfnl_Pro;

class Module extends Wpfnl_Frontend_Module
{
    public $stripe;
    public $offerStatus;
    public function __construct()
    {

        add_action( 'wp_enqueue_scripts', array( $this, 'load_offer_scripts' ) );

        /**
         * init upsell/downsell gateways
         * if user is in funnel
         */
        add_action( 'woocommerce_pre_payment_complete', array( $this, 'trigger_upsell_downsell_process' ), 99, 1 );

        /**
         * trigger the upsell/downsell process if this is not
         * triggered on previous hooks
         */
        add_action( 'woocommerce_checkout_order_processed', array( $this, 'maybe_trigger_upsell_downsell_process' ), 100, 3 );

        add_shortcode('funnel_sell_accept', [$this, 'funnel_upsell_shortcode_button']);
        add_shortcode('funnel_sell_reject', [$this, 'funnel_upsell_reject_shortcode_button']);

        add_action('wp_ajax_wpfnl_upsell_accepted', [$this, 'process_upsell_accepted_action']);
        add_action('wp_ajax_nopriv_wpfnl_upsell_accepted', [$this, 'process_upsell_accepted_action']);

        add_action('wp_ajax_wpfnl_upsell_rejected', [$this, 'process_upsell_rejected_action']);
        add_action('wp_ajax_nopriv_wpfnl_upsell_rejected', [$this, 'process_upsell_rejected_action']);


        add_action('wp_ajax_wpfnl_downsell_accepted', [$this, 'process_downsell_accepted_action']);
        add_action('wp_ajax_nopriv_wpfnl_downsell_accepted', [$this, 'process_downsell_accepted_action']);

        add_action('wp_ajax_wpfnl_downsell_rejected', [$this, 'process_downsell_rejected_action']);
        add_action('wp_ajax_nopriv_wpfnl_downsell_rejected', [$this, 'process_downsell_rejected_action']);

        add_action('wp_ajax_wpfnl_after_process_with_auth_ajax', [$this, 'wpfnl_after_process_with_auth_ajax']);
        add_action('wp_ajax_nopriv_wpfnl_after_process_with_auth_ajax', [$this, 'wpfnl_after_process_with_auth_ajax']);


        //== Discount price support ==//
        add_filter('wpfnl_offer_custom_price', [$this, 'wpfnl_modify_product_price'], 10, 1);
        add_filter('wpfnl_offer_product_price', [$this, 'wpfnl_modify_product_price'], 10, 1);

        wp_ajax_helper()->handle('wpfnl-get-variation-id')
            ->with_callback([ $this, 'wpfnl_get_variation_id' ]);
          

    }


    /**
     * trigger upsell/downsell process
     *
     * @param $order_id
     * @param $posted_data
     * @param $order
     *
     * @since 1.0.0
     */
    public function maybe_trigger_upsell_downsell_process($order_id, $posted_data, $order) {
    }


    /**
     * setup
     *
     * @param string $order_id
     *
     * @since 1.0.0
     */
    public function trigger_upsell_downsell_process( $order_id = '' ) {

        if ( '' == $order_id ) {
            return;
        }
        $order = wc_get_order( $order_id );
        $this->start_upsell_downsell_process( $order );
    }


    /**
     * start the upsell/downsell process
     *
     * @param $order
     *
     * @since 1.0.0
     */
    private function start_upsell_downsell_process($order) {
        if ( ! is_object( $order ) ) {
            return;
        }

        $order_gateway = $order->get_payment_method();
        $payment_gateway_obj = Wpfnl_Pro::instance()->payment_gateways->build_gateway($order_gateway);
    }


    /**
     * enqueue upsell/downsell offer
     * scripts
     *
     * @since 1.0.0
     */
    public function load_offer_scripts() {
        if( Wpfnl_functions::is_wc_active() && Wpfnl_Pro_functions::is_upsell_downsell_step() ) {
            global $post;
            $step_id                    = $post->ID;
            $funnel_id                  = get_post_meta( $step_id, '_funnel_id', true );
            $step_type                  = get_post_meta( $step_id, '_step_type', true );
            $order_id                   = ( isset( $_GET['wpfnl-order'] ) ) ? intval( $_GET['wpfnl-order'] ) : 0;
            $order_key                  = ( isset( $_GET['wpfnl-key'] ) ) ? sanitize_text_field( wp_unslash( $_GET['wpfnl-key'] ) ) : '';
            $order                      = wc_get_order( $order_id );
            $payment_method             = '';
            $is_reference_transaction   = false;
            $skip_offer                 = 'no';
            $offer_product              = Wpfnl_Pro_functions::get_offer_product_data( $step_id, '', '', $order_id );
          
            $quantity                   = 0;
            $product_id                 = '';


            if ( is_array($offer_product) && !empty($offer_product)) {
                $product_id = $offer_product['id'];
                $quantity   = $offer_product['qty'];
            }
            $is_variable = false;
            if( $product_id ){
                $product = wc_get_product( $product_id );
                if( $product &&  $product->get_type() === 'variable' ){
                    $is_variable = true;
                }
            }

            if($order) {
                $gateways = array( 'paypal','ppec_paypal' );
                $payment_method = $order->get_payment_method();
                // for now we ignore the paypal reference transactions. In future there will be a settings
                // for paypal reference transaction. User can define if they want to use paypal reference transaction
                // or not.
                if ( ( in_array( $payment_method, $gateways ) ) && !$is_reference_transaction ) {
                    $skip_offer = 'yes';
                }
            }
            $currency_symbol    = get_woocommerce_currency_symbol();
         
            $localize_data = array(
                'ajaxUrl'                       => admin_url('admin-ajax.php'),
                'step_id'                       => $step_id,
                'funnel_id'                     => $funnel_id,
                'is_lms'                        => get_post_meta($funnel_id, '_wpfnl_funnel_type', true) === 'lms' ? 'yes' : 'no',
                'product_id'                    => $product_id,
                'is_variable'                   => $is_variable,
                'quantity'                      => $quantity,
                'order_id'                      => $order_id,
                'order_key'                     => $order_key,
                'offer_type'                    => get_post_meta($step_id, '_step_type', true),
                'currency_symbol'               => $currency_symbol,
                'skip_offer'                    => $skip_offer,
                'wpfnl_offer_nonce'             => wp_create_nonce( 'wpfnl_offer_nonce' ),
                'wpfnl_upsell_accepted_nonce'   => wp_create_nonce( 'wpfnl_upsell_accepted_nonce' ),
                'wpfnl_upsell_rejected_nonce'   => wp_create_nonce( 'wpfnl_upsell_rejected_nonce' ),
                'wpfnl_downsell_accepted_nonce' => wp_create_nonce( 'wpfnl_downsell_accepted_nonce' ),
                'wpfnl_downsell_rejected_nonce' => wp_create_nonce( 'wpfnl_downsell_rejected_nonce' ),
                'wpfnl_create_paypal_order_nonce'=> wp_create_nonce( 'wpfnl_create_paypal_order_nonce' ),
                'wpfnl_capture_paypal_order_nonce'=> wp_create_nonce( 'wpfnl_capture_paypal_order_nonce' ),
                'payment_gateway'               => $payment_method,
            );

            if ( 'stripe' === $payment_method ) {
                $localize_data['wpfnl_stripe_sca_check_nonce'] = wp_create_nonce( 'wpfnl_stripe_sca_check_nonce' );
                wp_register_script( 'stripe', 'https://js.stripe.com/v3/', '', '3.0', true );
                wp_enqueue_script( 'stripe' );
            }

            if ( 'woocommerce_payments' === $payment_method ) {
				$localize_data['wpfunnels_woop_create_payment_intent'] = wp_create_nonce( 'wpfunnels_woop_create_payment_intent' );
				wp_register_script( 'stripe', 'https://js.stripe.com/v3/', '', '3.0', true );
				wp_enqueue_script( 'stripe' );
			}
            
            if ( 'mollie_wc_gateway_creditcard' === $payment_method ) {
                $localize_data['wpfnl_mollie_cc_process_nonce'] = wp_create_nonce( 'wpfnl_mollie_cc_process_nonce' );
            }

            if ( 'mollie_wc_gateway_creditcard' === $payment_method ) {
                $localize_data['wpfnl_mollie_ideal_process_nonce'] = wp_create_nonce( 'wpfnl_mollie_ideal_process_nonce' );
            }

            $localize_script  = '<!-- script to print WPFunnels Offer variables -->';
            $localize_script .= '<script type="text/javascript">';
            $localize_script .= 'var WPFunnelsOfferVars = ' . wp_json_encode( $localize_data ) . ';';
            $localize_script .= '</script>';
            echo $localize_script;
        }else{

            global $post;
            $step_id                    = '';
            if( $post ){
                $step_id                    = $post->ID;
            }
           
            $funnel_id                  = get_post_meta( $step_id, '_funnel_id', true );
            $step_type                  = get_post_meta( $step_id, '_step_type', true );

            $localize_data = array(
                'ajaxUrl'                       => admin_url('admin-ajax.php'),
                'step_id'                       => $step_id,
                'funnel_id'                     => $funnel_id,
                'is_lms'                        => get_post_meta($funnel_id, '_wpfnl_funnel_type', true) === 'lms' ? 'yes' : 'no',
                'wpfnl_offer_nonce'             => wp_create_nonce( 'wpfnl_offer_nonce' ),
                'wpfnl_upsell_accepted_nonce'   => wp_create_nonce( 'wpfnl_upsell_accepted_nonce' ),
                'wpfnl_upsell_rejected_nonce'   => wp_create_nonce( 'wpfnl_upsell_rejected_nonce' ),
                'wpfnl_downsell_accepted_nonce' => wp_create_nonce( 'wpfnl_downsell_accepted_nonce' ),
                'wpfnl_downsell_rejected_nonce' => wp_create_nonce( 'wpfnl_downsell_rejected_nonce' ),
                'wpfnl_create_paypal_order_nonce'=> wp_create_nonce( 'wpfnl_create_paypal_order_nonce' ),
                'wpfnl_capture_paypal_order_nonce'=> wp_create_nonce( 'wpfnl_capture_paypal_order_nonce' ),
            );

            $localize_script  = '<!-- script to print WPFunnels Offer variables -->';
            $localize_script .= '<script type="text/javascript">';
            $localize_script .= 'var WPFunnelsOfferVars = ' . wp_json_encode( $localize_data ) . ';';
            $localize_script .= '</script>';
            echo $localize_script;
        }
    }


    /**
     * Process upsell accept ajax action
     *
     * @since 1.0.0
     */
    public function process_upsell_accepted_action() {
       
        $nonce = filter_input( INPUT_POST, 'security', FILTER_SANITIZE_STRING );

        if ( $nonce && ! wp_verify_nonce( $nonce, 'wpfnl_upsell_accepted_nonce' ) ) {
            return array(
                'status'    => 'success',
                'message'   => __( 'Unauthorized request', 'wpfnl-pro' )
            );
        }
        
        
        
        $step_id        = isset( $_POST['step_id'] ) ? intval( $_POST['step_id'] ) : 0;
        $funnel_id      = isset( $_POST['funnel_id'] ) ? intval( $_POST['funnel_id'] ) : 0;
        $order_id       = isset( $_POST['order_id'] ) ? intval( $_POST['order_id'] ) : 0;
        $order_key      = isset( $_POST['order_key'] ) ? sanitize_text_field( wp_unslash($_POST['order_key']) ) : '';
        $offer_type     = isset( $_POST['offer_type'] ) ? sanitize_text_field( $_POST['offer_type'] ) : '';
        $offer_action   = isset( $_POST['offer_action'] ) ? sanitize_text_field( $_POST['offer_action'] ) : '';
        $product_id     = isset( $_POST['product_id'] ) ? intval( $_POST['product_id'] ) : 0;
        $quantity       = isset( $_POST['quantity'] ) ? intval( $_POST['quantity'] ) : 0;
        $variation_id       = isset( $_POST['variation_id'] ) ? intval( $_POST['variation_id'] ) : 0;
        $step_type      = get_post_meta($step_id, '_step_type', true);
        $attr = [];
        if( isset( $_POST['is_variable'], $_POST['attr'] ) && $_POST['is_variable'] == 'true' ){
            $payload = [
                'product_id' => $product_id,
                'data'       => $_POST['attr'],
            ]; 
            $attr = $_POST['attr'];
            
            $product_id = $this->wpfnl_get_variation_id($payload);
        }
        if( $variation_id ){
            $product_id = $variation_id;
        }

        $order = wc_get_order($order_id);

        $wpfnl_functions_instance = new Wpfnl_functions();
        if( method_exists($wpfnl_functions_instance,'is_valid_order_owner')){
            if (!Wpfnl_functions::is_valid_order_owner($order)) {
                wp_send_json([
                    'status'  => 'error',
                    'message' => __( 'Unauthorized request', 'wpfnl-pro' ),
                ]);
            }
        }
        
        $all_products   = [];
        $product_data   = array();
        if ( $order_id && $product_id ) {
            $data = array(
                'order_id'      => $order_id,
                'product_id'    => $product_id,
                'quantity'      => $quantity,
                'order_key'     => $order_key,
                'step_type'     => $step_type,
                'step_id'       => $step_id,
                'funnel_id'     => $funnel_id,
            );
           
            $result =  $this->offer_accepted( $step_id, $data , $attr );
            wp_send_json($result);
        } else {
            wp_send_json([
                'status' => 'error',
                'message' => 'No product data found',
            ]);
        }
    }



    /**
     * Process downsell accept ajax action
     *
     * @since 1.0.0
     */
    public function process_downsell_accepted_action() {

        $nonce = filter_input( INPUT_POST, 'security', FILTER_SANITIZE_STRING );
        if ( $nonce && ! wp_verify_nonce( $nonce, 'wpfnl_downsell_accepted_nonce' ) ) {
            return array(
                'status'    => 'success',
                'message'   => __( 'Unauthorized request', 'wpfnl-pro' )
            );
        }

        $step_id        = isset( $_POST['step_id'] ) ? intval( $_POST['step_id'] ) : 0;
        $funnel_id      = isset( $_POST['funnel_id'] ) ? intval( $_POST['funnel_id'] ) : 0;
        $order_id       = isset( $_POST['order_id'] ) ? intval( $_POST['order_id'] ) : 0;
        $order_key      = isset( $_POST['order_key'] ) ? sanitize_text_field( wp_unslash($_POST['order_key']) ) : '';
        $offer_type     = isset( $_POST['offer_type'] ) ? sanitize_text_field( $_POST['offer_type'] ) : '';
        $offer_action   = isset( $_POST['offer_action'] ) ? sanitize_text_field( $_POST['offer_action'] ) : '';
        $product_id     = isset( $_POST['product_id'] ) ? intval( $_POST['product_id'] ) : 0;
        $quantity       = isset( $_POST['quantity'] ) ? intval( $_POST['quantity'] ) : 0;
        $variation_id   = isset( $_POST['variation_id'] ) ? intval( $_POST['variation_id'] ) : 0;
        $step_type      = get_post_meta($step_id, '_step_type', true);
        $attr = [];
        if( isset( $_POST['is_variable'], $_POST['attr'] ) && $_POST['is_variable'] == 'true' ){
            $payload = [
                'product_id' => $product_id,
                'data'       => $_POST['attr'],
            ]; 
            $attr = $_POST['attr'];
            
            $product_id = $this->wpfnl_get_variation_id($payload);
        }

        if( $variation_id ){
            $product_id = $variation_id;
        }

        $order = wc_get_order($order_id);

        $wpfnl_functions_instance = new Wpfnl_functions();
        if( method_exists($wpfnl_functions_instance,'is_valid_order_owner')){
            if (!Wpfnl_functions::is_valid_order_owner($order)) {
                wp_send_json([
                    'status'  => 'error',
                    'message' => __( 'Unauthorized request', 'wpfnl-pro' ),
                ]);
            }
        }

        $all_products   = [];
        $product_data   = array();

        if ( $order_id && $product_id ) {

            $data = array(
                'order_id'      => $order_id,
                'product_id'    => $product_id,
                'quantity'      => $quantity,
                'order_key'     => $order_key,
                'step_type'     => $step_type,
                'step_id'       => $step_id,
                'funnel_id'     => $funnel_id,
            );
            $result =  $this->offer_accepted( $step_id, $data , $attr );

            wp_send_json($result);
        } else {
            wp_send_json([
                'status' => 'error',
                'message' => 'No product data found',
            ]);
        }
    }



    /**
     * Process upsell reject ajax action
     *
     * @since 1.0.0
     */
    public function process_upsell_rejected_action() {
        $nonce = filter_input( INPUT_POST, 'security', FILTER_SANITIZE_STRING );
        if ( ! wp_verify_nonce( $nonce, 'wpfnl_upsell_rejected_nonce' ) ) {
            return array(
                'status'    => 'success',
                'message'   => __( 'Unauthorized request', 'wpfnl-pro' )
            );
        }
      
        $step_id        = isset( $_POST['step_id'] ) ? intval( $_POST['step_id'] ) : 0;
        $funnel_id      = isset( $_POST['funnel_id'] ) ? intval( $_POST['funnel_id'] ) : 0;
        $order_id       = isset( $_POST['order_id'] ) ? intval( $_POST['order_id'] ) : 0;
        $order_key      = isset( $_POST['order_key'] ) ? sanitize_text_field( wp_unslash($_POST['order_key']) ) : '';
        $product_id     = isset( $_POST['product_id'] ) ? intval( $_POST['product_id'] ) : 0;
        $quantity       = isset( $_POST['quantity'] ) ? intval( $_POST['quantity'] ) : 0;
        $step_type      = get_post_meta($step_id, '_step_type', true);

        $result = array(
            'status'   => 'failed',
            'redirect' => '#',
        );

        if ( $step_id ) {

            $type = get_post_meta( $funnel_id, '_wpfnl_funnel_type', true );
            if( $type === 'lms' ){

                /** Change lms order meta value */
                Wpfnl_lms_learndash_functions::update_lms_order_status( $funnel_id, $step_id, get_current_user_id() );
        
                $result['message'] = 'Redirecting...';
                $result['status'] = 'success';
                $result['redirect_url'] = $this->offer_rejected( $step_id, [] );
            }else{
                $result = array(
                    'status'        => 'failed',
                    'redirect_url'  => '#',
                    'message'       => __( 'Order does not exist', 'wpfnl-pro' ),
                );
    
                if ( $order_id ) {
                    $data = array(
                        'order_id'      => $order_id,
                        'product_id'    => $product_id,
                        'quantity'      => $quantity,
                        'order_key'     => $order_key,
                        'step_type'     => $step_type,
                        'step_id'       => $step_id,
                        'funnel_id'     => $funnel_id,
                    );
                    $result['message'] = 'Redirecting...';
                    $result['status'] = 'success';
                    $result['redirect_url'] = $this->offer_rejected( $step_id, $data );
                }
            }
            
        }
        
        // send json.
        wp_send_json( $result );
    }


    /**
     * Process downsell reject ajax action
     *
     * @since 1.0.0
     */
    public function process_downsell_rejected_action() {
        $nonce = filter_input( INPUT_POST, 'security', FILTER_SANITIZE_STRING );
        if ( ! wp_verify_nonce( $nonce, 'wpfnl_downsell_rejected_nonce' ) ) {
            return array(
                'status'    => 'success',
                'message'   => __( 'Unauthorized request', 'wpfnl-pro' )
            );
        }
        $step_id        = isset( $_POST['step_id'] ) ? intval( $_POST['step_id'] ) : 0;
        $funnel_id      = isset( $_POST['funnel_id'] ) ? intval( $_POST['funnel_id'] ) : 0;
        $order_id       = isset( $_POST['order_id'] ) ? intval( $_POST['order_id'] ) : 0;
        $order_key      = isset( $_POST['order_key'] ) ? sanitize_text_field( wp_unslash($_POST['order_key']) ) : '';
        $product_id     = isset( $_POST['product_id'] ) ? intval( $_POST['product_id'] ) : 0;
        $quantity       = isset( $_POST['quantity'] ) ? intval( $_POST['quantity'] ) : 0;
        $step_type      = get_post_meta($step_id, '_step_type', true);

        $result = array(
            'status'   => 'failed',
            'redirect' => '#',
        );

        if ( $step_id ) {
            $type = get_post_meta( $funnel_id, '_wpfnl_funnel_type', true );
            if( $type === 'lms' ){

                /** Change lms order meta value */
                Wpfnl_lms_learndash_functions::update_lms_order_status( $funnel_id, $step_id, get_current_user_id() );

                $result['message'] = 'Redirecting...';
                $result['status'] = 'success';
                $result['redirect_url'] = $this->offer_rejected( $step_id, [] );
            }else{

                $result = array(
                    'status'        => 'failed',
                    'redirect_url'  => '#',
                    'message'       => __( 'Order does not exist', 'wpfnl-pro' ),
                );

                if ( $order_id ) {
                    $data = array(
                        'order_id'      => $order_id,
                        'product_id'    => $product_id,
                        'quantity'      => $quantity,
                        'order_key'     => $order_key,
                        'step_type'     => $step_type,
                        'step_id'       => $step_id,
                        'funnel_id'     => $funnel_id,
                    );
                    $result['message']      = 'Redirecting...';
                    $result['status']       = 'success';
                    $result['redirect_url'] = $this->offer_rejected( $step_id, $data );
                }
            }
        }
        
        // send json.
        wp_send_json( $result );
    }


    /**
     * Process upsell accept ajax action with auth
     *
     * @since 1.0.0
     */
    public function wpfnl_after_process_with_auth_ajax()
    {
        $step_id = $_POST['step_id'];
        $order_id = $_POST['order_id'];
        $order_key = $_POST['order_key'];
        $step_type = get_post_meta($step_id, '_step_type', true);
        $intent_id = '';

        if (isset($_POST['intent_id'])) {
            $intent_id = $_POST['intent_id'];
        }

        $extra_data = [];
        $extra_data['step_id'] = $step_id;
        $extra_data['step_type'] = $step_type;
        $extra_data['order_id'] = $order_id;
        $extra_data['order_key'] = $order_key;
        $extra_data['intent_id'] = $intent_id;
        
        $step_meta = get_post_meta($step_id, '_wpfnl_'.$step_type.'_products', true);

        if ($step_meta[0]) {
            $step_meta = $step_meta[0];
        } else {
            wp_send_json([
                'status' => 'error',
                'message' => 'No attach product found',
            ]);
        }

        $product_id = $step_meta['id'];
        $quantity = $step_meta['quantity'];
        $extra_data['product_id'] = $product_id;
        $extra_data['input_qty'] = $quantity;
        
        $response = $this->offer_accepted($step_id, $extra_data);
        wp_send_json($response);
    }


    /**
     * offer accepted action
     *
     * @param $step_id
     * @param $data
     * @return array
     * @throws \Exception
     */
    public function offer_accepted( $step_id, $data , $attr = null) {
        $order_id               = $data['order_id'];
        $funnel_id              = $data['funnel_id'];
        $order_key              = $data['order_key'];
        $input_qty              = $data['quantity'];
        $order                  = wc_get_order($order_id);
        $skip_payment           = filter_input(INPUT_POST, 'stripe_sca_payment', FILTER_VALIDATE_BOOLEAN);
        $order_gateway          = $order->get_payment_method();
        if( 'stripe_cc' ===  $order_gateway ){
            $order_gateway = 'stripe';
        }
        $payment_gateway_obj    = Wpfnl_Pro::instance()->payment_gateways->build_gateway($order_gateway);
        $is_charge_success      = false;
        
        if ($skip_payment) {
            $_stripe_intent_id = get_post_meta($order->get_id(), '_stripe_intent_id_' . $step_id, true);
            $intent_id = filter_input(INPUT_POST, 'stripe_intent_id', FILTER_SANITIZE_STRING);
            $skip_payment = ($intent_id === $_stripe_intent_id) ? true : false;
        }
        
        $offer_product = Wpfnl_Pro_functions::update_variation_product_details( $step_id, $data['product_id'], $input_qty, $order_id  );
       
        $product       = wc_get_product($data['product_id']);

        if( $product && !$product->is_in_stock() || $product->managing_stock() && ( $product->get_stock_quantity() < intval($offer_product['qty']) || $product->get_stock_quantity() == 0 || intval($offer_product['qty']) == 0 ) ) {
            if( 'variation' === $product->get_type() ) {
                $message = __( 'The selected variation is out of stock. Please, try choosing different variation.', 'wpfnl-pro' );
            }
            else {
                $message = __( 'Product is out of stock', 'wpfnl-pro' );
            }
            return array(
                'status'   => 'failed',
                'message'  => $message,
            );
        }
        $no_payment_gateway = false;
        $shipping = [];
       
        if($offer_product['qty'] != 0){
            if ( (isset($offer_product['price']) && '' === trim($offer_product['price'])) || $skip_payment ) {
                $is_charge_success = array(
                    'is_success'    => true,
                    'message'       => '',
                );
                
                if ( $skip_payment ) {
                    
                    $response       = \WC_Stripe_API::retrieve('charges?payment_intent=' . $intent_id);
                    $charge_data    = reset($response->data);
                    $order->update_meta_data('_wpfunnels_offer_txn_resp_' . $step_id, $charge_data->id);
                    $order->save();
                }
            } else {
                if ($payment_gateway_obj) {
                    $tax_enabled            = get_option( 'woocommerce_calc_taxes' );
                    if ( 'yes' === $tax_enabled ) {
                        if ( !wc_prices_include_tax() ) {
                            $product = wc_get_product($offer_product['id']);
                            $offer_product['total'] = $product ? wc_get_price_including_tax( $product, array( 'price' => $offer_product['total'] ) ) : '';
                        }
                    }
                    
                    $offer_product_id         = isset( $offer_product[ 'id' ] ) ? $offer_product[ 'id' ] : 0;
                    $offer_total_price        = isset( $offer_product[ 'total' ] ) ? $offer_product[ 'total' ] : 0;
                    $shipping                 = $this->calculate_offer_product_shipping_price( $order, $offer_product_id, $offer_total_price );
                    $shipping_price           = isset( $shipping[ 'total' ] ) ? $shipping[ 'total' ] : 0;
                    $shipping_tax             = isset( $shipping[ 'total_tax' ] ) ? $shipping[ 'total_tax' ] : 0;
                    $shipping_price           = $shipping_price + $shipping_tax;
                    
                    $offer_product[ 'price' ] = $offer_total_price + $shipping_price;
                    $offer_product[ 'total' ] = $offer_total_price + $shipping_price;
                   
                    $offer_product[ 'price' ] = $offer_total_price;
                    $offer_product[ 'total' ] = $offer_total_price;

                    $chained_product_class_instance = new \WPFunnelsPro\Compatibility\ChainedProduct();
                    $total_chained_product_price = 0;
                    if( Wpfnl_functions::is_wc_active() ){
                        $chained_products = $chained_product_class_instance->get_chain_product_details( $offer_product['id'] );
                        
                        if ( !empty($chained_products) ){
                            foreach ($chained_products as $chained_product_id => $chianed_product){
                                $new_product_price = 0; 
                                $product = wc_get_product( $chained_product_id );
                                $chained_product = wc_get_product( $chained_product_id );
                                if( 'yes' === $chianed_product['priced_individually'] ){
                                    $chained_product_price = $chained_product->get_price();
                                    if ( ! empty( $chained_product_price ) ) {
                                        $total_chained_product_price += $chained_product_price * $chianed_product['unit'];
                                    }
                                }
                            }
                        }
                    }
                    $offer_product[ 'price' ] = $offer_product[ 'price' ] + $total_chained_product_price;
                    $offer_product[ 'total' ] = $offer_product[ 'total' ] + $total_chained_product_price;

                    $is_charge_success        = $payment_gateway_obj->process_payment( $order, $offer_product );
                  

                }else{
                    $no_payment_gateway = true;
                }
            }
        }
        
        if( isset($is_charge_success['is_success'] ) && $is_charge_success['is_success'] ) {
            $response           = Wpfnl_Pro_functions::after_offer_charged( $funnel_id, $step_id, $order_id, $order_key, $offer_product, true , $attr, '', '', $shipping );
            if( 'success' === $response['status'] ) {
                delete_option( 'wpfunnels_dynamic_offer_data' );
                $result = array(
                    'status'   => 'success',
                    'message'  => __( 'Offer product is added successfully', 'wpfnl-pro' ),
                );
            } else {
                $result = array(
                    'status'   => 'success',
                    'message'  => __( 'Offer product is not added', 'wpfnl-pro' ),
                );
            }
        } else{
            if( $no_payment_gateway ){
                $result = array(
                    'status'   => 'failed',
                    'message'  => __( 'No payment gateway found', 'wpfnl-pro' ),
                );
            }else{
                $result = array(
                    'status'   => 'failed',
                    'message'  => $is_charge_success['message'],
                );
            }
            
        }
        $this->offerStatus      = 'accept';

        $result['redirect_url'] = $this->get_next_step_url( $order, $step_id, $funnel_id );
        $next_step  = Wpfnl_functions::get_next_step( $funnel_id, $step_id );
        $next_step  = apply_filters( 'wpfunnels/next_step_data', $next_step );
        $custom_url = Wpfnl_functions::custom_url_for_thankyou_page( $next_step[ 'step_id' ] );
       
        if( $custom_url ){
            $result['redirect_url'] = $custom_url;
        }

        if( isset($result['redirect_url']) && !$result['redirect_url'] ){
            $result['redirect_url'] = Wpfnl_functions::redirect_to_deafult_thankyou();
        }
        
        return $result;
    }


    /**
     * @param $step_id
     * @param $data
     * @return string
     */
    public function offer_rejected( $step_id, $data ) {

        if( $data ){
            $order              = wc_get_order($data['order_id']);
            $product_id         = $data['product_id'];
            $input_qty          = $data['quantity'];
            $this->offerStatus  = 'reject';
            $offer_product      = Wpfnl_Pro_functions::get_offer_product_data( $step_id, $product_id, $input_qty, $data['order_id'] );
            delete_option( 'wpfunnels_dynamic_offer_data' );
            do_action( 'wpfunnels/offer_rejected', $order, $offer_product );

            $funnel_id  = Wpfnl_functions::get_funnel_id_from_step( $step_id );
            $next_step  = Wpfnl_functions::get_next_step( $funnel_id, $step_id );
            $next_step  = apply_filters( 'wpfunnels/next_step_data', $next_step );
            $custom_url = Wpfnl_functions::custom_url_for_thankyou_page( $next_step[ 'step_id' ] );

            if( $order ){
                $order->update_meta_data('_wpfunnels_order', 'yes');
                $order->update_meta_data('_wpfunnels_funnel_id', $funnel_id);
            }

            if( $custom_url ){
                return $custom_url;
            }
            if( $this->get_next_step_url( $order, $step_id, $funnel_id ) ){
                return $this->get_next_step_url( $order, $step_id, $funnel_id );
            }
            
            return Wpfnl_functions::redirect_to_deafult_thankyou();

        }else{
            $funnel_id     = Wpfnl_functions::get_funnel_id_from_step( $step_id );
            $step_type     = get_post_meta( $step_id, '_step_type', true );
            $offer_product = get_post_meta( $step_id, '_wpfnl_' . $step_type . '_products', true );

            if( isset($offer_product[0]) ){
                $offer_product[0]['step_id'] = $step_id;
                do_action( 'wpfunnels/offer_rejected', '', $offer_product[0] );
            }

            $next_step  = Wpfnl_functions::get_next_step( $funnel_id, $step_id );
            $next_step  = apply_filters( 'wpfunnels/next_step_data', $next_step );
            $custom_url = Wpfnl_functions::custom_url_for_thankyou_page( $next_step[ 'step_id' ] );

            if( $custom_url ){
                return $custom_url;
            }
            return $this->get_next_step_url_for_lms( (object)[], $step_id, $funnel_id );
        }
        
    }


    /**
     * get next step url
     *
     * @param \WC_Order $order
     * @param $step_id
     * @param $funnel_id
     * @return string
     */
    public function get_next_step_url_for_lms( $order , $step_id, $funnel_id ) {
        $current_page_id = $step_id;
        $next_step       = Wpfnl_functions::get_next_step( $funnel_id, $current_page_id );
        $next_step       = apply_filters( 'wpfunnels/next_step_data', $next_step );

        if ( 'conditional' === $next_step['step_type'] ) {
            $condition_identifier = strval($next_step['step_id']);
            $condition_matched  = Wpfnl_functions::check_condition($funnel_id, $order, $condition_identifier, $current_page_id, $this->offerStatus);
            $node_found         = Wpfnl_functions::get_node_id($funnel_id, $next_step['step_id']);
            if ($condition_matched) {
                $next_node = $this->go_to_output_1($funnel_id, $node_found);
                $next_step = Wpfnl_functions::get_step_by_node($funnel_id, $next_node);
            } else {
                $next_node = $this->go_to_output_2($funnel_id, $node_found);
                $next_step = Wpfnl_functions::get_step_by_node($funnel_id, $next_node);
            }
        } else {
            $next_step = $next_step['step_id'];
        }
        $redirect_url = get_post_permalink($next_step);

        
        $query_args = array(
            'wpfnl-order' => 'lms_order',
            'wpfnl-key' => 'lms_order',
        );
    
        
        return add_query_arg($query_args, $redirect_url);
    }


    /**
     * get next step url
     *
     * @param \WC_Order $order
     * @param $step_id
     * @param $funnel_id
     * @return string
     */
    public function get_next_step_url( \WC_Order $order , $step_id, $funnel_id ) {
        $current_page_id = $step_id;
        $next_step       = Wpfnl_functions::get_next_step( $funnel_id, $current_page_id );
        $next_step       = apply_filters( 'wpfunnels/next_step_data', $next_step );
        
        if ( isset($next_step['step_type']) && 'conditional' === $next_step['step_type'] ) {
            $condition_identifier = strval($next_step['step_id']);
            $condition_matched  = Wpfnl_functions::check_condition($funnel_id, $order, $condition_identifier, $current_page_id, $this->offerStatus);
            $node_found         = Wpfnl_functions::get_node_id($funnel_id, $next_step['step_id']);
            if ($condition_matched) {
                $next_node = $this->go_to_output_1($funnel_id, $node_found);
                $next_step = Wpfnl_functions::get_step_by_node($funnel_id, $next_node);
                $next_step_data = [ 'step_id' => $next_step, 'step_type' => get_post_meta( $next_step, '_step_type', true) ];
                $next_step       = apply_filters( 'wpfunnels/next_step_data', $next_step_data );
                $next_step_id       = isset( $next_step[ 'step_id' ] ) ? $next_step[ 'step_id' ] : '';
            } else {
                $next_node = $this->go_to_output_2($funnel_id, $node_found);
                $next_step = Wpfnl_functions::get_step_by_node($funnel_id, $next_node);
                $next_step_data = [ 'step_id' => $next_step, 'step_type' => get_post_meta( $next_step, '_step_type', true) ];
                $next_step       = apply_filters( 'wpfunnels/next_step_data', $next_step_data );
                $next_step_id       = isset( $next_step[ 'step_id' ] ) ? $next_step[ 'step_id' ] : '';
            }
        }else {
            $next_step_id = isset($next_step['step_id']) ? $next_step['step_id'] : '';
        }
        
        if( $order ){
            $next_step              = apply_filters( 'wpfunnels/modify_next_step_based_on_order', $next_step, $order );
            $next_step_id = isset($next_step['step_id']) ? $next_step['step_id'] : '';
        }
       
        $redirect_url = $next_step_id  ? get_post_permalink($next_step_id) : '';
        if( $order ){
            $query_args = array(
                'wpfnl-order' => $order->get_id(),
                'wpfnl-key' => $order->get_order_key(),
            );
        }else{
            $query_args = array(
                'wpfnl-order' => 'lms_order',
                'wpfnl-key' => 'lms_order',
            );
        }
        if( $redirect_url ){
            return add_query_arg($query_args, $redirect_url);
        }
        return false;
        
    }



    public function detect_upsell() {
        $type = '';
        if (get_post_type(get_the_ID()) == 'wpfunnel_steps') {
            $type = get_post_meta(get_the_ID(), '_step_type', true);
            if ($type == 'upsell' || $type == 'downsell') {
                if (isset($_GET['wpfnl-order']) && isset($_GET['wpfnl-key'])) {
                    return;
                } else {
                    $user = wp_get_current_user();
                    if (!in_array('administrator', (array)$user->roles)) {
                        echo '<p style="color:red;text-align:center;">You are not allowed to access this page directly</p>';
                        wp_die();
                    }
                }
            }
        }
    }


    public function wpfnl_modify_product_price($product_price) {
        if (class_exists('WOOMC\MultiCurrency\API')) {
            $api_obj = new WOOMC\MultiCurrency\API();
            $product_price = $api_obj->convert($product_price, get_woocommerce_currency(), get_option('woocommerce_currency'));
        }

        return $product_price;
    }

    public function funnel_upsell_shortcode_button() {
        global $post;
        return '
            <div class="wpfnl-upsell-btn-wrapper">
                <button type="button" id="upsell_funnel_target" data-id="' . $post->ID . '" name="button">
                    BUY OFFER
                    <span class="wpfnl-loader" id="wpfnl-loader-accept"></span>
                </button>
                <span class="wpfnl-alert" id="wpfnl-alert-accept"></span>
            </div>
        ';
        //wpfnl-success, wpfnl-error, box
    }

    public function funnel_upsell_reject_shortcode_button() {
        global $post;
        return '
            <div class="wpfnl-upsell-btn-wrapper">
                <button type="button" id="upsell_funnel_reject" data-id="' . $post->ID . '" name="button">
                    Not Interested
                    <span class="wpfnl-loader" id="wpfnl-loader-reject"></span>
                </button>
                <span class="wpfnl-alert" id="wpfnl-alert-reject"></span>
            </div>
        ';
        //wpfnl-success, wpfnl-error, box
    }

    public function go_to_output_1($funnel_id, $node_found) {
        $funnel_json = get_post_meta($funnel_id, 'funnel_data', true);
        if ($funnel_json) {
            // $funnel_data = json_decode($funnel_json, true);
            $node_data = $funnel_json['drawflow']['Home']['data'];

            foreach ($node_data as $node_key => $node_value) {
                if ($node_value['id'] == $node_found) {
                    $next_node = isset($node_value['outputs']['output_1']['connections'][0]['node']) ? $node_value['outputs']['output_1']['connections'][0]['node'] : '';
                    return $next_node;
                }
            }
            return false;
        }
    }

    public function go_to_output_2($funnel_id, $node_found) {
        $funnel_json = get_post_meta($funnel_id, 'funnel_data', true);
        if ($funnel_json) {
            // $funnel_data = json_decode($funnel_json, true);
            $node_data = $funnel_json['drawflow']['Home']['data'];

            foreach ($node_data as $node_key => $node_value) {
                if ($node_value['id'] == $node_found) {
                    $next_node = isset($node_value['outputs']['output_2']['connections'][0]['node']) ? $node_value['outputs']['output_2']['connections'][0]['node'] : '';
                    return $next_node;
                }
            }
            return false;
        }
    }

    public function get_conditional_redirect($step_id, $response, $type = 'accept') {
        $funnel_id = '';
        $condition_identifier = '';
        $group_conditions = get_post_meta($funnel_id, $condition_identifier, true);
        if ($group_conditions) {

            // Loop through group condition.
            foreach ($group_conditions as $group) {
                if (empty($group)) {
                    continue;
                }

                $match_group = true;
                // Loop over rules and determine if all rules match.
                foreach ($group as $rule) {
                    if (!$this->match_rule($rule, $order, $current_page_id)) {
                        $match_group = false;
                        break;
                    }
                }

                // If this group matches, show the field group.
                if ($match_group) {
                    return true;
                }
            }
        }

        // Return default.
        return false;

    }

    public function get_discount_data($step_id) {
        $data = [];
        $type = get_post_meta($step_id, '_step_type', true);
        if ($type == 'upsell') {
            $data['type'] = get_post_meta($step_id, '_wpfnl_upsell_discount_type', true);
            $data['value'] = get_post_meta($step_id, '_wpfnl_upsell_discount_value', true);
        } else {
            $data['type'] = get_post_meta($step_id, '_wpfnl_downsell_discount_type', true);
            $data['value'] = get_post_meta($step_id, '_wpfnl_downsell_discount_value', true);
        }
        return $data;
    }


    /**
     * 
     */
    private function wpfnl_get_variation_id( $payload ){
        if( isset($payload['product_id'], $payload['data']) && is_array($payload['data']) ){
            $variation_id = (new \WC_Product_Data_Store_CPT())->find_matching_product_variation(
                    new \WC_Product($payload['product_id']),
                    $payload['data']
            );
            return $variation_id;
        }
        return false;
    }


    /**
     * @desc calculate shipping cost for offer steps (upsell/downsell)
     *
     * @param \WC_Order $order
     * @param $product_id
     * @param $total_price
     * @return int|mixed|void
     */
    private function calculate_offer_product_shipping_price( \WC_Order $order, $product_id, $total_price ) {
        if( !$product_id ) {
            return;
        }
        if( !WC()->shipping()->get_shipping_classes() ) {
            return;
        }

        $product = wc_get_product( $product_id );
        if( $product && $product->is_virtual() ) {
            return;
        }

        $shipping            = [];
        $shipping_methods    = $order->get_shipping_methods();
        $offer_settings      = Wpfnl_functions::get_offer_settings();
        $offer_orders_option = isset( $offer_settings[ 'offer_orders' ] ) ? $offer_settings[ 'offer_orders' ] : '';

        if( !empty( $shipping_methods ) ) {
            foreach( $shipping_methods as $key => $method ) {
                if( $method ){
                    $shipping = [
                        'name'           => $method->get_name(),
                        'method_title'   => $method->get_method_title(),
                        'method_id'      => $method->get_method_id(),
                        'instance_id'    => $method->get_instance_id(),
                        'total'          => $method->get_total(),
                        'total_tax'      => $method->get_total_tax(),
                        'taxes'          => $method->get_taxes(),
                        'tax_percentage' => ( $method->get_total() != 0 ) || ( $method->get_total() != 0.00 ) ? ( $method->get_total_tax() * 100 ) / $method->get_total() : 0,
                    ];
                }
                
            }
        }

        if( 'main-order' === $offer_orders_option ) {
            $wc_shipping_zones   = \WC_Shipping_Zones::get_zones();

            if( !empty( $wc_shipping_zones ) && isset( $shipping[ 'method_id' ] ) && 'flat_rate' === $shipping[ 'method_id' ] ) {
                foreach( $wc_shipping_zones as $zone ) {
                    if( isset( $zone[ 'shipping_methods' ] ) && !empty( $zone[ 'shipping_methods' ] ) ) {
                        foreach( $zone[ 'shipping_methods' ] as $method ) {
                            if( 'WC_Shipping_Flat_Rate' === get_class( $method ) ) {
                                if( isset( $method->instance_id ) && isset( $shipping[ 'instance_id' ] ) ) {
                                    if( $method->instance_id == $shipping[ 'instance_id' ] ) {
                                        $product_shipping_class_id = $product->get_shipping_class_id();
                                        $instance_settings         = isset( $method->instance_settings ) ? $method->instance_settings : [];
                                        $shipping_cost             = isset( $instance_settings[ 'class_cost_' . $product_shipping_class_id ] ) ? $instance_settings[ 'class_cost_' . $product_shipping_class_id ] : $instance_settings[ 'no_class_cost' ];
                                        $shipping_cost             = $shipping_cost ? $shipping_cost : 0; 
                                        $tax_percentage          = isset( $shipping[ 'tax_percentage' ] ) ? $shipping[ 'tax_percentage' ] : 0;
                                        $total_tax               = ( $shipping_cost * $tax_percentage ) / 100;
                                        $shipping[ 'total' ]     = $shipping_cost;
                                        $shipping[ 'total_tax' ] = $total_tax;
                                        $shipping[ 'taxes' ]     = [
                                            'total' => [ $total_tax ]
                                        ];
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        unset( $shipping[ 'tax_percentage' ] );
        return $shipping;
    }
}
