<?php
/**
 * This class is responsible for all the helper functions
 * 
 * @package WPFunnels
 */
namespace WPFunnels;

class Wpfnl_functions
{
	public static $installed_plugins;

	/**
	 * Get all the steps of the funnel
	 *
	 * @param $funnel_id
	 * 
	 * @return array|mixed
	 */
	public static function get_steps( $funnel_id ) {
		$steps = get_post_meta( $funnel_id, '_steps_order', true );
		if ( ! is_array( $steps ) ) {
			$steps = array();
		}
		return $steps;
	}

	
	/**
	 * Check if the associated order is from a funnel.
	 *
	 * This function checks if the given order is associated with a funnel.
	 * It returns true if the order is from a funnel, and false otherwise.
	 * Previously 'get_funnel_id_from_order' function parameter was order id but from 2.8.0 version it changed to order object
	 * 
	 * @param \WC_Order $order The WooCommerce order object.
	 * @return bool True if the order is from a funnel, false otherwise.
	 * @since 2.8.0
	 */
	public static function check_if_funnel_order($order) {
		if ( false === is_a( $order, 'WC_Order' ) ) {
            return false;
        }
		$funnel_id = self::get_funnel_id_from_order($order);
		if( !$funnel_id ){
			return false;
		}
		return true;
	}


	/**
	 * Get the associated funnel ID from the order ID.
	 *
	 * This function retrieves the ID of the associated funnel for a given order.
	 * Previously data was retrieved by post meta table but from 2.8.0 version user get_meta function to retrieve meta
	 *
	 * @param \WC_Order|Int $order The WooCommerce order object or Order ID of WooCommerce order.
	 * @return int|false The ID of the associated funnel, or false if not found.
	 * @since 2.8.0
	 */
	public static function get_funnel_id_from_order($order) {
		if( !$order ){
			return false;
		}

		if ( false === is_a( $order, 'WC_Order' ) ) {
			$order_id = intval($order);
			$order = wc_get_order( $order_id );
            if ( false === is_a( $order, 'WC_Order' ) ) {
				return false;
			}
        }

		$funnel_id = $order->get_meta('_wpfunnels_funnel_id') ? $order->get_meta('_wpfunnels_funnel_id') : $order->get_meta('_wpfunnels_parent_funnel_id');
		if (!$funnel_id) {
			return false;
		}
		return intval($funnel_id);
	}

	/**
	 * Check if the given string is a date or not
	 *
	 * @param $date
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
	public static function validate_date($date)
	{
		return (bool)strtotime($date);
	}

	/**
	 * Define constant if it is not set yet
	 *
	 * @param $name
	 * @param $value
	 *
	 * @since 2.0.3
	 */
	public static function maybe_define_constant( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	/**
	 * Set do not cache constants
	 */
	public static function do_not_cache() {
		global $post;

		if ( ! apply_filters( 'wpfunnels/do_not_cache', true, $post->ID ) ) {
			return;
		}
		self::maybe_define_constant( 'DONOTCACHEPAGE', true );
		self::maybe_define_constant( 'DONOTCACHEOBJECT', true );
		self::maybe_define_constant( 'DONOTCACHEDB', true );
		nocache_headers();
	}

	/**
	 * Return formatted date from the
	 * given date object
	 *
	 * @param $date
	 * 
	 * @return false|string
	 * @since  1.0.0
	 */
	public static function get_formatted_date($date)
	{
		return date('Y-m-d h:i A', strtotime($date));
	}


	/**
	 * Get checkout ID
	 * 
	 * @return bool|int
	 */
    public static function get_checkout_id_from_post_data() {
        $checkout_id = !empty( $_POST[ '_wpfunnels_checkout_id' ] ) ? filter_var( wp_unslash( $_POST[ '_wpfunnels_checkout_id' ] ), FILTER_SANITIZE_NUMBER_INT ) : false;
        return $checkout_id ? intval( $checkout_id ) : false;
    }


	/**
	 * Get checkout id from post
	 *
	 * @return bool|int
	 */
	public static function get_checkout_id_from_post( $post = null )
	{
		if( isset($post['post_data']) ){
			$post_data  = array();
			parse_str($post['post_data'], $post_data);
			if( isset($post_data['_wpfunnels_checkout_id']) ){
				return $post_data['_wpfunnels_checkout_id'];
			}
		}
		return false;
	}


	/**
	 * Check order bump clicked or not from post data
	 * 
	 * @return bool True|False
	 * @since 2.8.0
	 */
    public static function is_orderbump_clicked_from_post_data() {
        if( !empty( $_POST[ 'post_data' ] ) ) {
            wp_parse_str( $_POST[ 'post_data' ], $post_data );
            return !empty( $post_data[ '_wpfunnels_order_bump_clicked' ] ) && 'yes' === $post_data[ '_wpfunnels_order_bump_clicked' ];
        }
        return !empty( $_POST[ '_wpfunnels_order_bump_clicked' ] ) && 'yes' === $_POST[ '_wpfunnels_order_bump_clicked' ];
    }


	/**
	 * Check variation is selected or not from post data
	 * 
	 * @return bool True|False
	 * @since 2.8.0
	 */
    public static function is_variation_selected_from_post_data() {
        if( !empty( $_POST[ 'post_data' ] ) ) {
            wp_parse_str( $_POST[ 'post_data' ], $post_data );
            return !empty( $post_data[ '_wpfunnels_variable_product' ] ) && 'selected' === $post_data[ '_wpfunnels_variable_product' ];
        }
        return !empty( $_POST[ '_wpfunnels_variable_product' ] ) && 'selected' === $_POST[ '_wpfunnels_variable_product' ];
    }
	

	/**
	 * Get funnel id
	 *
	 * @param $step_id
	 * 
	 * @return mixed
	 */
	public static function get_funnel_id_from_step($step_id) {
		$funnel_id = get_post_meta($step_id, '_funnel_id', true);
		return intval($funnel_id);
	}

    /**
     * Get order id from funnel post data
     *
     * @return false|int
     * @since 2.8.2
     */
    public static function get_order_id_from_post_data() {
        if( !empty( $_POST[ '_wpfunnels_order_unique_identifier' ] ) ) {
            global $wpdb;
            $identifier    = wp_unslash( $_POST[ '_wpfunnels_order_unique_identifier' ] );
            $prepare_guery = $wpdb->prepare( "SELECT `post_id` FROM {$wpdb->postmeta} WHERE meta_key ='_wpfunnels_order_unique_identifier' and meta_value like '%s'", $identifier );
            $get_value     = $wpdb->get_row( $prepare_guery );
            return $get_value ? intval( $get_value->post_id ) : false;
        }
        return false;
    }


	/**
	 * Get checkout id from order
	 * 
	 * @param $order_id
	 * 
	 * @return int
	 */
	public static function get_checkout_id_from_order( $order_id ) {
		$checkout_id = get_post_meta( $order_id, '_wpfunnels_checkout_id', true );
		return $checkout_id ? intval( $checkout_id ) : false;
	}


	/**
	 * Get funnel from post data
	 * 
	 * @return bool|int
	 */
    public static function get_funnel_id_from_post_data() {
        if( !empty( $_POST[ '_wpfunnels_funnel_id' ] ) ) {
            $funnel_id = filter_var( wp_unslash( $_POST[ '_wpfunnels_funnel_id' ] ), FILTER_SANITIZE_NUMBER_INT );
            return $funnel_id ? intval( $funnel_id ) : false;
        }
        return false;
    }


	/**
	 * Get funnel id from step page
	 *
	 * @return false|mixed
	 */
	public static function get_funnel_id() {
		global $post;
		if ( $post ) {
			$funnel_id = get_post_meta( $post->ID, '_funnel_id', true );
            return $funnel_id ? intval( $funnel_id ) : false;
		}
		return false;
	}

	/**
	 * Unserialize data
	 *
	 * @param $data
	 * 
	 * @return mixed
	 * @since  1.0.0
	 */
	public static function unserialize_array_data($data)
	{
        return maybe_unserialize($data);
	}


	/**
	 * Get formatted string with phrase
	 * e.g: 1 item if singular or
	 * 2 items if plural
	 *
	 * @param Int $number
	 * @param String $singular
	 * @param String $plural
	 * 
	 * @return String
	 * @since  1.0.0
	 */
    public static function get_formatted_data_with_phrase( $number, $singular = '', $plural = 's' ) {
        return ( is_numeric( $number ) && 0 <= $number ) ? ( 1 < $number ? $plural : $singular ) : null;
    }


	/**
	 * Get formatted string from funnel status
	 *
	 * @param $status
	 * 
	 * @return string
	 * @since  1.0.0
	 */
    public static function get_formatted_status( $status ) {
        switch( $status ) {
            case 'publish':
                return 'Published';
            case 'draft':
                return 'Draft';
            default:
                return $status;
        }
    }


	/**
	 * Generate active class for funnel menus
	 *
	 * @param $key
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
	public static function define_active_class($key)
	{
		$page = filter_input(INPUT_GET, 'page', FILTER_SANITIZE_SPECIAL_CHARS);
		if ($page === WPFNL_MAIN_PAGE_SLUG && $key === 'overview') {
			return true;
		}

		if ($page === 'create_funnel' && $key === 'create_funnel') {
			return true;
		}

		if ($page === 'wpfnl_settings' && $key === 'settings') {
			return true;
		}

		if ($page === 'wpfunnels_integrations' && $key === 'integrations') {
			return true;
		}
	}


	/**
	 * Check If module exists or not
	 * 
	 * @param $id
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
    public static function check_if_module_exists( $id ) {
        return 'publish' === get_post_status( $id ) || 'draft' === get_post_status( $id );
    }


	/**
	 * Check if the cpt is step or not
	 *
	 * @param $step_id
	 * @param string $type
	 * 
	 * @return bool
	 */
	public static function check_if_this_is_step_type_by_id($step_id, $type = 'landing')
	{
		$post_type = get_post_type($step_id);
		if (WPFNL_STEPS_POST_TYPE === $post_type) {
			if ($type === get_post_meta($step_id, '_step_type', true)) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Get thankyou page id
	 * 
	 * @param $funnel_id
	 * 
	 * @return bool|mixed
	 *
	 * @since 2.2.6
	 */
    public static function get_thankyou_page_id( $funnel_id ) {
        $steps = self::get_steps( $funnel_id );
        if( is_array( $steps ) ) {
            foreach( $steps as $step ) {
                if( 'thankyou' === $step[ 'step_type' ] ) {
                    return $step[ 'id' ];
                }
            }
        }
        return false;
    }


	/**
	 * Check if the current post type is step or not
	 *
	 * @return bool
	 * @since  2.0.3
	 */
	public static function is_funnel_step_page( $post_type = '' )
	{
		if( self::get_current_post_type($post_type) === WPFNL_STEPS_POST_TYPE ) {
			return true;
		}

        $post_id = null;
        if ( wp_doing_ajax() || isset( $_GET['wc-ajax'] ) ) {
            $post_id = self::get_checkout_id_from_post($_POST);
        }
        global $post;
        $post = $post_id ? get_post( $post_id ) : $post;
        $post_type = $post->post_type ?? '';
        if( self::get_current_post_type($post_type) === WPFNL_STEPS_POST_TYPE ){
            return true;
        }
		return false;
	}


	/**
	 * Get current post type
	 *
	 * @param $post_type
	 * 
	 * @return string
	 *
	 * @since 2.3.5
	 */
	public static function get_current_post_type( $post_type ) {
		global $post;
		if ( '' === $post_type && is_object( $post ) ) {
			$post_type = $post->post_type;
		}else{
			$post_id = '';
			if ( wp_doing_ajax() || isset( $_GET['wc-ajax'] ) ) {
				$post_id = self::get_checkout_id_from_post($_POST);
			}

			if( $post_id ){
				$post = get_post( $post_id );
				$post_type = $post->post_type;
			}
		}
		return $post_type;
	}


	/**
	 * Check if funnel checkout page
	 *
	 * @param $funnel_id
	 * 
	 * @return bool
	 */
	public static function is_funnel_checkout_page()
	{
		if(isset($_POST['post_data'])){
			parse_str($_POST['post_data'],$post_data);
			if(isset($post_data['_wpfunnels_checkout_id'])){
				return [
					'status' => true,
					'id'	 => $post_data['_wpfunnels_checkout_id']
				];
			}
		}
		return [
			'status' => false,
			'id'	 => ''
		];
	}


	/**
	 * Check if funnel exists
	 *
	 * @param $funnel_id
	 * 
	 * @return bool
	 */
	public static function is_funnel_exists($funnel_id)
	{
		if (!$funnel_id) return false;
		if (FALSE === get_post_status($funnel_id)) {
			return true;
		}
		return false;
	}


	/**
	 * Function to check if the current page is a post edit page
	 *
	 * @return bool
	 *
	 * @since 2.0.3
	 */
	public static function is_step_edit_page(){

		$step_id = -1;
		if ( is_admin() && isset( $_REQUEST['action'] ) ) {
			if ( 'edit' === $_REQUEST['action'] && isset( $_GET['post'] ) ) {
				$step_id = isset( $_GET[ 'post' ] ) ? $_GET[ 'post' ] : - 1;
			}
			elseif ( isset( $_REQUEST['wpfunnels_gb'] ) && isset( $_POST['post_id'] ) ){ //phpcs:ignore
				$step_id = intval( $_POST['post_id'] ); //phpcs:ignore
			}
			if ( $step_id === - 1 ) {

				return false;
			}
			$get_post_type = get_post_type( $step_id );
			if ( WPFNL_STEPS_POST_TYPE === $get_post_type ) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Check if the cpt is step or not
	 *
	 * @param string $type
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
	public static function check_if_this_is_step_type($type = 'landing')
	{
		$post_type = get_post_type();

		if (WPFNL_STEPS_POST_TYPE === $post_type) {
			global $post;
			if ($type === get_post_meta($post->ID, '_step_type', true)) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Hooks for start and end the journey
	 * of a funnel
	 */
	public static function start_journey() {
		$post_type = get_post_type();
		if ( WPFNL_STEPS_POST_TYPE === $post_type ) {
			global $post;
			$step_id 		= $post->ID;
			$funnel_id 		= self::get_funnel_id_from_step($step_id);
			$steps 			= self::get_steps($funnel_id);
			$funnel_data 	= self::get_funnel_data($funnel_id);
			// start the journey
			if( $steps && is_array($steps) ) {
				if( $steps[0]['id'] === $step_id ) {
					do_action( 'wpfunnels/funnel_journey_starts', $step_id, $funnel_id );
				}
			}

			// end the journey
			if( $funnel_data && isset($funnel_data['drawflow']['Home']['data']) ) {
				$funnel_data = $funnel_data['drawflow']['Home']['data'];
				foreach ( $funnel_data as $data ) {
					$info 		= $data['data'];
					$step_type 	= $info['step_type'];

					if('conditional' !== $step_type) {
						$_step_id 	= isset($info['step_id']) ? $info['step_id'] : '';
						$output_con = $data['outputs'];

						if( empty($output_con) ) {
							if( $_step_id === $step_id ) {
								do_action( 'wpfunnels/funnel_journey_end', $step_id, $funnel_id );
								delete_option( 'optin_data' );
								break;
							}
						}
					}
				}
			}
		}
	}


	/**
	 * Conditional node logic check
	 *
	 * @param $funnel_id
	 * @param $order
	 * @param $condition_identifier
	 * @param $current_page_id
	 * @param $checker
	 * 
	 * @return bool
	 *
	 * @since 2.0.2
	 */
	public static function check_condition( $funnel_id, $order, $condition_identifier, $current_page_id, $checker = 'accept' )
	{

		$group_conditions = get_post_meta( $funnel_id, $condition_identifier, true );

		if ($group_conditions) {
			// Loop through group condition.
			foreach ($group_conditions as $group) {
				if (empty($group)) {
					continue;
				}

				$match_group = true;
				// Loop over rules and determine if all rules match.
				foreach ($group as $rule) {
					if (!self::match_rule( $funnel_id, $rule, $order, $current_page_id, $checker )) {
						$match_group = false;
						break;
					}
				}

				// If this group matches, show the field group.
				if ($match_group) {
					return true;
				}else{
					return false;
				}
			}
		}
		if( $checker == 'accept' ){
			return true;
		}
		// Return default.
		return false;
	}


	/**
	 * Check if rule is matched
	 *
	 * @param $rule
	 * @param $order
	 * @param $current_page_id
	 * @param $checker
	 * 
	 * @return mixed
	 *
	 * @since 2.0.2
	 */
	public static function match_rule( $funnel_id, $rule, $order, $current_page_id, $checker )
	{

		if (isset( $rule['field'] ) && $rule['field'] == 'downsell') {
			$rule['field'] = 'upsell';
		}
		$condition = $rule['field'];
		if ( strpos($rule['field'] , 'optin')){
			$condition = 'optin';
		}
		$checker_function = $condition . '_condition_checker';
		return self::$checker_function( $funnel_id, $rule, $order, $current_page_id, $checker );
	}
	

	/**
	 * Upsell condition checker for conditional step
  *
	 * @param Integer $funnel_id
	 * @param Array $data
	 * @param Object $order
	 * @param Integer $current_page_id
	 * @param String $checker
	 * 
	 * @return Boolean
	 * 
	 * @since 2.5.7
	 */
	public static function upsell_condition_checker( $funnel_id ,$data, $order, $current_page_id, $checker)
	{

		if ($data['value'] == 'yes') {
			if ( $checker == 'accept' ) {
				return true;
			} else {
				return false;
			}

		} else if ($data['value'] == 'no') {
			
			if ($checker == 'reject') {
				return true;
			} else {
				return false;
			}
		}
		return false;
	}


	/**
	 * Optin condition checker for conditional step
  *
	 * @param Integer $funnel_id
	 * @param Array $data
	 * @param Object $order
	 * @param Integer $current_page_id
	 * @param String $checker
	 * 
	 * @return Boolean
	 * 
	 * @since 2.5.7
	 */
	public function optin_condition_checker( $funnel_id, $rule, $data, $current_page_id, $checker = 'accept' ){

		if( isset($data['step_id']) ){
			$step_id = str_replace('optin_','',$data['step_id'] );
			$optin_steps = get_option('optin_data');
			if( isset( $optin_steps['optin_data'] )){
				$key = array_search($step_id, array_column($optin_steps['optin_data'], 'step_id'));
				if( false !== $key ){
					unset($optin_steps[$key]);
					update_option('optin_data', $optin_steps );
					return true;
				}
			}

		}elseif(!empty($data['cta']) ){
			return false;
		}

		if( isset($rule['field']) ){
			$step_id = str_replace('optin_','',$rule['field'] );
			$optin_steps = get_option('optin_data');
			if( isset( $optin_steps['optin_data'] )){
				$key = array_search($step_id, array_column($optin_steps['optin_data'], 'step_id'));
				if( false !== $key ){
					unset($optin_steps[$key]);
					update_option('optin_data', $optin_steps );
					return true;
				}
			}
		}

		return false;
	}


	/**
	 * Order bump condition checker
	 * 
	 * @param $data
	 * @param $order
	 * @param $current_page_id
	 * 
	 * @return bool
	 */
	public static function orderbump_condition_checker( $funnel_id, $data, $order, $current_page_id, $checker = 'accept' )
	{
		$order_bump_accepted = '';
		if ( Wpfnl_functions::is_wc_active() ) {
			$order_bump_accepted = WC()->session->get('order_bump_accepted');
		}
		return $data['value'] == $order_bump_accepted;
	}


	/**
	 * Check cart total condition in conditional node
	 * 
	 * @param $data
	 * @param $order
	 * @param $current_page_id
	 * 
	 * @return bool
	 */
	public static function carttotal_condition_checker($funnel_id, $data, $order, $current_page_id, $checker = 'accept' )
	{
		$cart_total = $order->get_total();

		$checker = false;
		if ($data['condition'] == 'greater') {
			if ($cart_total > $data['value']) {
				$checker = true;
			}
		} elseif ($data['condition'] == 'equal') {
			if ($cart_total == $data['value']) {
				$checker = true;
			}
		} elseif ($data['condition'] == 'less') {
			if ($cart_total < $data['value']) {
				$checker = true;
			}
		}
		return $checker;
	}


	/**
	 * Billing country condition checker
	 *
	 * @param Array
	 * @param Object
	 * @param String
	 * @param String
	 *
	 * @return Bool
	 * @since  2.4.18
	 */
	public static function billing_condition_checker($funnel_id, $data, $order, $current_page_id, $checker ){
		if( $order ){
			$billing_country = $order->get_billing_country();
			if( isset($data['field']) && 'billing' === $data['field'] && isset($data['value']) && $billing_country === $data['value'] ){
				return true;
			}
		}
		return false;
	}


	/**
	 * Shipping country condition checker
	 *
	 * @param Array
	 * @param Object
	 * @param String
	 * @param String
	 *
	 * @return Bool
	 * @since  2.4.18
	 */
	public static function shipping_condition_checker( $funnel_id, $data, $order, $current_page_id, $checker ){
		if( $order ){
			$shipping_country = $order->get_shipping_country();
			if( isset($data['field']) && 'shipping' === $data['field'] && isset($data['value']) && $shipping_country === isset($data['value']) ){
				return true;
			}
		}
		return false;
	}


	/**
	 * Get next step of the
	 * funnel
	 *
	 * @param $funnel_id
	 * @param $step_id
	 * @param bool $condition
	 * 
	 * @return Mix
	 *
	 * @since 1.0.0
	 */
	public static function get_next_step( $funnel_id, $step_id, $condition = true )
	{
		if( $funnel_id && !$step_id ) {
			return false;
		}
		$parent_step_id =  get_post_meta( $step_id, '_parent_step_id', true );
		if( $parent_step_id ){
			$step_id = $parent_step_id;
		}
		$funnel_data = self::get_funnel_data($funnel_id);
		if ( $funnel_data ) {
			$node_id        = self::get_node_id( $funnel_id, $step_id );

			$node_data      = $funnel_data['drawflow']['Home']['data'];
			$step_type =  get_post_meta( $step_id, '_step_type', true );
			foreach ( $node_data as $node_key => $node_value ) {
				if ( $node_value['id'] == $node_id ) {
					$triggers = self::get_mint_triggers();
					if( in_array( $step_type, $triggers ) ){
						$next_node_id 	= isset($node_value['outputs']['output_1']['connections'][0]['node']) ? $node_value['outputs']['output_1']['connections'][0]['node'] : '';
					}else{
						if( $condition ) {
							$next_node_id 	= isset($node_value['outputs']['output_1']['connections'][0]['node']) ? $node_value['outputs']['output_1']['connections'][0]['node'] : '';
						} else {
							$next_node_id 	= isset($node_value['outputs']['output_2']['connections'][0]['node']) ? $node_value['outputs']['output_2']['connections'][0]['node'] : '';
						}
					}
					
					if( $next_node_id ){
						$next_step_id 	= self::get_step_by_node( $funnel_id, $next_node_id );
						$next_step_type = self::get_node_type( $node_data, $next_node_id );
						if( in_array( $next_step_type, $triggers ) ){
							$next_step = self::get_next_step( $funnel_id, $next_step_id );
							return $next_step;
						}
						
						return apply_filters( 'wpfunnels/next_step_data', array(
							'step_id' 	=> $next_step_id,
							'step_type' => $next_step_type,
						)); 
						
					}else{
						// if there is no thank you page
						self::redirect_to_deafult_thankyou();
					}
				}
			}
		}
		return false;
	}


	/**
	 * Redirect deafult thank you page
	 */
	public static function redirect_to_deafult_thankyou(){

		if( isset( $_POST[ 'order_id' ] ) && $_POST[ 'order_id' ] ) {
			$url = home_url().'/checkout/order-received/'.$_POST[ 'order_id' ].'/?key='.$_POST[ 'order_key' ];
			return $url;
		}else{
			add_action( 'template_redirect', function( $order_id ){
				if( isset( $_GET[ 'order-received' ] ) && $_GET[ 'order-received' ] ) {
					$url = '';
					$url = add_query_arg( 'id', $_GET[ 'order-received' ], $url );
					wp_safe_redirect(  wp_sanitize_redirect( esc_url_raw( $url ) ) );
					exit;
				}elseif( isset( $_POST[ 'order_id' ] ) && $_POST[ 'order_id' ] ) {
					$url = '';
					$url = add_query_arg( 'id', $_POST[ 'order_id' ], $url );
					wp_safe_redirect(  wp_sanitize_redirect( esc_url_raw( $url ) ) );
					exit;
				}
			});
		}
	}


	/**
	 * Get previous step of the
	 * funnel
	 *
	 * @param $funnel_id
	 * @param $step_id
	 * @param bool $condition
	 * 
	 * @return Mix
	 *
	 * @since 1.0.0
	 */
	public static function get_prev_step( $funnel_id, $step_id, $condition = true )
	{
		if( $funnel_id && !$step_id ) {
			return false;
		}
		$funnel_data = self::get_funnel_data($funnel_id);
		if ( $funnel_data ) {
			$node_id        = self::get_node_id( $funnel_id, $step_id );
			$node_data      = $funnel_data['drawflow']['Home']['data'];

			foreach ( $node_data as $node_key => $node_value ) {
				if ( $node_value['id'] == $node_id ) {
					if( $condition ) {
						if(!empty($node_value['inputs'])){
							$prev_node_id 	= $node_value['inputs']['input_1']['connections'][0]['node'];
						}else{
							$prev_node_id 	= '';
						}

					} else {
						if(!empty($node_value['inputs'])){
							$prev_node_id 	= $node_value['inputs']['input_2']['connections'][0]['node'];
						}else{
							$prev_node_id 	= '';
						}

					}
					$prev_step_id 	= self::get_step_by_node( $funnel_id, $prev_node_id );
					$prev_step_type = self::get_node_type( $node_data, $prev_node_id );
					if($prev_step_type == 'conditional'){
						return self::get_prev_step($funnel_id,$prev_step_id);

					}else{

						return array(
							'step_id' 	=> $prev_step_id,
							'step_type' => $prev_step_type,
						);
					}
				}
			}

		}
		return false;
	}

	/**
	 * Get node type
	 */
	public static function get_node_type($node_data, $node_id)
	{
		foreach ($node_data as $node_key => $node_value) {
			if ($node_value['id'] == $node_id) {
				return $node_value['data']['step_type'];
			}
		}
	}

	/**
	 * Get step by node
	 */
	public static function get_step_by_node($funnel_id, $node_id)
	{
		$identifier_json = get_post_meta($funnel_id, 'funnel_identifier', true);
		$identifier_json = preg_replace('/\: *([0-9]+\.?[0-9e+\-]*)/', ':"\\1"', $identifier_json);
		if ($identifier_json) {
			$identifier = json_decode($identifier_json, true);
			foreach ($identifier as $identifier_key => $identifier_value) {
				if ($identifier_key == $node_id) {
					return $identifier_value;
				}
			}
		}
		return false;
	}


	/**
	 * Get node by step
	 *
	 * @param $funnel_id
	 * @param $step_id
	 * 
	 * @return bool|int|string
	 */
	public static function get_node_id( $funnel_id, $step_id )
	{
		$identifier_json = get_post_meta( $funnel_id, 'funnel_identifier', true );
		if ($identifier_json) {
			$identifier = json_decode( $identifier_json, true );
			foreach ( $identifier as $identifier_key => $identifier_value ) {
				if ($identifier_value == $step_id) {
					return $identifier_key;
				}
			}
		}
		return false;
	}


	/**
	 * Get funnel data
	 *
	 * @param $funnel_id
	 * 
	 * @return mixed
	 *
	 * @since 2.0.5
	 */
	public static function get_funnel_data( $funnel_id ) {
		return get_post_meta( $funnel_id, 'funnel_data', true );
	}


	/**
	 * Update settings option
	 *
	 * @param $key
	 * @param $value
	 * @param bool $network
	 * 
	 * @since 1.0.0
	 */
	public static function update_admin_settings($key, $value, $network = false)
	{
		if ( $network && is_multisite() ) {
			update_site_option($key, $value);
		} else {
			update_option($key, $value);
		}
	}


	/**
	 * Get admin settings option
	 * by key
	 *
	 * @param $key
	 * @param bool $default
	 * @param bool $network
	 * 
	 * @return mixed|void
	 * @since  1.0.0
	 */
	public static function get_admin_settings($key, $default = false, $network = false)
	{
		if ($network && is_multisite()) {
			$value = get_site_option($key, $default);
		} else {
			$value = get_option($key, $default);
		}
		return $value;
	}


	/**
	 * Get general settings data
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public static function get_general_settings()
	{
		$default_settings = apply_filters(
			'wpfunnels_general_settings',
			[
				'builder' => 'elementor',
				'paypal_reference' => 'off',
				'order_bump' => 'off',
				'ab_testing' => 'off',
				'allow_funnels' => Array
				(
					'administrator' => 'true'
				),
				'funnel_type' => 'lead',
				'create_child_order' => 'off',
				'disable_theme_style' => 'off',
				'enable_log_status' => 'off',
			]
		);
		$saved_settings = self::get_admin_settings('_wpfunnels_general_settings', $default_settings);
		return wp_parse_args($saved_settings, $default_settings);
	}


	/**
	 * Get permalink settings data
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public static function get_permalink_settings()
	{
		$default_settings = apply_filters(
			'wpfunnels_permalink_settings',
			[
				'structure' => 'default',
				'step_base' => WPFNL_STEPS_POST_TYPE,
				'funnel_base' => WPFNL_FUNNELS_POST_TYPE,
			]
		);
		$saved_settings = self::get_admin_settings('_wpfunnels_permalink_settings');
		return wp_parse_args($saved_settings, $default_settings);
	}


	/**
	 * Get offer settings data
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public static function get_offer_settings()
	{
		$default_settings = apply_filters( 'wpfunnels/get_offer_settings', array(
				'offer_orders' => 'main-order',
				'show_supported_payment_gateway' => 'off',
				'skip_offer_step' => 'off',
				'skip_offer_step_for_free' => 'off',
			)
		);
		$saved_settings = self::get_admin_settings('_wpfunnels_offer_settings');
		return wp_parse_args($saved_settings, $default_settings);
	}

	/**
	 * Get GTM events
	 *
	 * @return array
	 */
	public static function get_gtm_events(){
		$default_gtm_events = array(
			'add_to_cart' 		=> 'Add to cart',
			'begin_checkout' 	=> 'Begin checkout',
			'add_payment_info' 	=> 'Add Payment Info',
			'add_shipping_info' => 'Add Shipping Info',
			'purchase' 			=> 'Purchase',
			'orderbump_accept' 	=> 'Order Bump',
			'upsell' 			=> 'Upsell',
			'downsell' 			=> 'Downsell',
		);
		return $default_gtm_events;
	}

	public static function get_gtm_settings(){
		$default_enable_settings = array(
			'gtm_enable'		=> 'off',
			'gtm_container_id' 	=> '',
			'gtm_events' 		=> array(
				'add_to_cart' 		=> 'true',
				'begin_checkout' 	=> 'true',
				'add_payment_info' 	=> 'true',
				'add_shipping_info' => 'true',
				'purchase' 			=> 'true',
				'orderbump_accept' 	=> 'true',
				'upsell' 			=> 'true',
				'downsell' 			=> 'true',
			),
		);
		$gtm_settings = self::get_admin_settings('_wpfunnels_gtm');
		return wp_parse_args($gtm_settings, $default_enable_settings);
	}
	/**
	 * Get facebook pixel events
	 *
	 * @return array
	 */
	public static function get_facebook_events(){
		$default_fb_events = array(
			'AddPaymentInfo' => 'Add payment info',
			'AddToCart' => 'Add to cart',
			'InitiateCheckout' => 'Initiate checkout',
			//'Lead' => 'Lead',
			'Purchase' => 'Purchase',
			'ViewContent' => 'View content',
		);
		return $default_fb_events;
	}

	public static function get_facebook_pixel_settings(){
		$default_enable_settings = array(
			'enable_fb_pixel'			 	=> 'off',
			'facebook_pixel_id' 		=> '',
			'facebook_tracking_events' 		=> array(
				'AddPaymentInfo' 	=> 'true',
				'AddToCart' 		=> 'true',
				'InitiateCheckout'  => 'true',
				'Lead' 				=> 'true',
				'Purchase' 			=> 'true',
				'ViewContent' 		=> 'true',
			),
		);
		$facebook_pixel_setting = self::get_admin_settings('_wpfunnels_facebook_pixel');
		return wp_parse_args($facebook_pixel_setting, $default_enable_settings);
	}


	public static function get_recaptcha_settings(){
		$default_enable_settings = array(
			'enable_recaptcha'			 	=> 'no',
			'recaptcha_site_key' 		=> '',
			'recaptcha_site_secret' 		=> '',
			);
		$recaptcha_setting = self::get_admin_settings('_wpfunnels_recaptcha_setting');
		return wp_parse_args($recaptcha_setting, $default_enable_settings);
	}


	/**
	 * Get advanced settings
	 */
	public static function get_advanced_settings(){
		$default_enable_settings = array(
			'show_supported_payment_gateway'	=> 'off',
		);

		$advanced_settings = self::get_admin_settings('_wpfunnels_advanced_settings');
		return wp_parse_args($advanced_settings, $default_enable_settings);
	}

	/**
	 * Get UTM Parameters
	 *
	 * @return array
	 */
	public static function get_utm_params(){
		$default_utm_params = array(
			'utm_source' 	=> 'UTM Source',
			'utm_medium' 	=> 'UTM Medium',
			'utm_campaign' 	=> 'UTM Campaign',
			'utm_content' 	=> 'UTM Content',
		);
		return $default_utm_params;
	}

	public static function get_utm_settings(){
		$default_enable_settings = array(
			'utm_enable'	=> 'off',
			'utm_source' 	=> '',
			'utm_medium' 	=> '',
			'utm_campaign' 	=> '',
			'utm_content' 	=> '',
		);
		$utm_settings = self::get_admin_settings('_wpfunnels_utm_params');
		return wp_parse_args($utm_settings, $default_enable_settings);
	}

	/**
	 * Get user roles
	 *
	 * @return array
	 * @since  2.1.7
	 */
	public static function get_user_roles(){
		global $wp_roles;
		$all_roles = $wp_roles->roles;
		return array_keys($all_roles);
	}


	/**
	 * Get the saved builder type
	 *
	 * @return mixed|void
	 * @since  1.0.0
	 */
	public static function get_builder_type()
	{
		$general_settings = self::get_general_settings();
		return $general_settings['builder'];
	}


	/**
	 * Check if wc is installed
	 *
	 * @return bool
	 * @since  1.0.0
	 */
    public static function is_wc_active()
    {
        if (in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            return true;
        }elseif( function_exists('is_plugin_active') ){
            if( is_plugin_active( 'woocommerce/woocommerce.php' )){
                return true;
            }
        }
        return false;
    }


	/**
	 * Check if elementor is installed
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_elementor_active()
	{
		if (in_array('elementor/elementor.php', apply_filters('active_plugins', get_option('active_plugins')))) {
			return true;
		}elseif( function_exists('is_plugin_active') ){
			if( is_plugin_active( 'elementor/elementor.php' )){
				return true;
			}
		}
		return false;
	}


	/**
	 * Check if saved builder is activated
	 * or not
	 *
	 * @param $builder
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_builder_active($builder)
	{
		switch ($builder) {
			case 'elementor':
				return self::is_elementor_active();
				break;
			default:
				return false;
				break;
		}
	}


	/**
	 * Check if the global funnel addon is activated or not
	 *
	 * @return bool
	 * @since  2.0.4
	 */
	public static function is_global_funnel_activated() {
		return apply_filters('wpfunnels/is_global_funnel_activated', false);
	}


	/**
	 * Check if the funnel is global funnel
	 *
	 * @param $funnel_id
	 * 
	 * @return bool
	 */
	public static function is_global_funnel( $funnel_id ) {
		if(!$funnel_id) {
			return false;
		}
		return apply_filters( 'wpfunnels/is_global_funnel', false, $funnel_id );
	}


	/**
	 * Check if pro is activated/deactivated
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_wpfnl_pro_activated()
	{
		return apply_filters('wpfunnels/is_wpfnl_pro_active', false) || apply_filters('is_wpfnl_pro_active', false);
	}


	/**
	 * Check if pro is activated/deactivated
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_pro_license_activated()
	{
		return apply_filters('wpfunnels/is_pro_license_activated', false);
	}


	/**
	 * Check if the module is pro or
	 * not
	 *
	 * @param $module
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_pro_module($module)
	{
		$pro_modules = apply_filters('wpfnl_pro_modules', []);
		return in_array($module, $pro_modules);
	}


	/**
	 * Check if the module is exists or not
	 *
	 * @param $module_name
	 * @param string $type
	 * @param bool $step
	 * @param bool $pro
	 * 
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_module_registered($module_name, $type = 'admin', $step = false, $pro = false)
	{
		$class_name = str_replace('-', ' ', $module_name);
		$class_name = str_replace(' ', '', ucwords($class_name));
		if ($pro) {
			if ($type === 'steps') {
				$class_name = 'WPFunnelsPro\\Admin\\Modules\\Steps\\' . $class_name . '\Module';
			}
		} else {
			if ($type === 'admin') {
				$class_name = 'WPFunnels\\Admin\\Modules\\' . $class_name . '\Module';
			} elseif ($type === 'steps') {
				$class_name = 'WPFunnels\\Admin\\Modules\\Steps\\' . $class_name . '\Module';
			} else {
				$class_name = 'WPFunnels\\Modules\\Frontend\\' . $class_name . '\Module';
			}
		}
		return class_exists($class_name);
	}


	/**
	 * Check manager permissions on REST API.
	 *
	 * @param string $object Object.
	 * @param string $context Request context.
	 * 
	 * @return bool
	 * @since  2.6.0
	 */
	public static function wpfnl_rest_check_manager_permissions($object, $context = 'read')
	{

		$objects = [
			'settings'      => 'manage_options',
			'templates'     => 'manage_options',
			'steps'         => 'manage_options',
			'products'      => 'manage_options',
		];
		return current_user_can( $objects[$object] );
	}


	/**
	 * Check if the provided plugin ($path) is installed or not
	 *
	 * @param $path
	 * 
	 * @return bool
	 * @since  2.0.0
	 */
	public static function is_plugin_installed( $path )
	{
		$plugins = get_plugins();
		return isset($plugins[$path]);
	}



	/**
	 * Check if the provided plugin ($path) is installed or not
	 *
	 * @param $path
	 * 
	 * @return bool
	 * @since  2.0.0
	 */
	public static function is_plugin_activated( $path )
	{
		include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		if(is_plugin_active( $path)) {
			return true;
		}
		return false;
	}



	/**
	 * Check plugin status by path
	 *
	 * @param $path
	 * @param $slug
	 * 
	 * @return string
	 *
	 * @since 2.0.1
	 */
	public static function get_plugin_action($path, $slug)
	{
		if( 'divi-builder' === $slug ){
			$is_divi_theme_active = self::wpfnl_is_theme_active( 'Divi' );
			if( $is_divi_theme_active ){
				return 'nothing';
			}
		}

		if (null == self::$installed_plugins) {
			self::$installed_plugins = get_plugins();
		}

		if (!isset(self::$installed_plugins[$path])) {
			return 'install';
		} elseif (!is_plugin_active($path)) {
			return 'activate';
		} else {
			return 'nothing';
		}
	}

	/**
	 * Check theme is active or not
	 *
	 * @param $theme_name
	 * 
	 * @return Bool
	 */
	public static function wpfnl_is_theme_active( $theme_name ){
        $theme = wp_get_theme(); // gets the current theme
        if ( $theme_name == $theme->name || $theme_name == $theme->parent_theme ) {
            return true;
        }
        return false;
    }


	/**
     * Check plugin is installed or not
     *
     * @param $plugin_slug
	 * 
     * @return Bolean
     */
    public static function wpfnl_check_is_plugin_installed( $plugin ){
        $installed_plugins = get_plugins();
        return array_key_exists( $plugin, $installed_plugins ) || in_array( $plugin, $installed_plugins, true );
    }


	/**
	 * Get depenedency plugins status
	 *
	 * @return mixed|void
	 *
	 * @since 2.0.1
	 */
	public static function get_dependency_plugins_status()
	{
		return apply_filters('wpfunnels/dependency_plugin_list', array(
			'elementor' => array(
				'name' 			=> 'Elementor',
				'plugin_file' 	=> 'elementor/elementor.php',
				'slug' 			=> 'elementor',
				'action' 		=> self::get_plugin_action('elementor/elementor.php', 'elementor')
			),
			'gutenberg' => array(
				'name' 			=> 'Qubely',
				'plugin_file' 	=> 'qubely/qubely.php',
				'slug' 			=> 'qubely',
				'action' 		=> self::get_plugin_action('qubely/qubely.php', 'qubely')
			),
			'divi-builder' => array(
				'name' 			=> 'Divi',
				'plugin_file' 	=> 'divi-builder/divi-builder.php',
				'slug' 			=> 'divi-builder',
				'action' 		=> self::get_plugin_action('divi-builder/divi-builder.php', 'divi-builder')
			),
			'oxygen' => array(
				'name' 			=> 'Oxygen',
				'plugin_file' 	=> 'oxygen/functions.php',
				'slug' 			=> 'oxygen',
				'action' 		=> self::get_plugin_action('oxygen/functions.php', 'oxygen')
			)
		));
	}


	/**
	 * Is there any missing plugin for wpfunnels
	 *
	 * @return string
	 *
	 * @since 2.0.1
	 */
	public static function is_any_plugin_missing()
	{
		if (null == self::$installed_plugins) {
			self::$installed_plugins = get_plugins();
		}
		$builder 			= self::get_builder_type();
		$dependency_plugins = self::get_dependency_plugins_status();
		$is_missing = 'no';

		if (isset($dependency_plugins[$builder])) {

			$plugin_data = $dependency_plugins[$builder];
			if ($plugin_data['action'] === 'activate' || $plugin_data['action'] === 'install') {
				$is_missing = 'yes';
			}
		}
		return $is_missing;
	}


	/**
	 * Recursively traverses a multidimensional array in search of a specific value and returns the
	 * array containing the value, or an
	 * null on failure.
	 *
	 * @param $search_value
	 * @param $array
	 * 
	 * @return array
	 * @since  2.0.0
	 */
	public static function recursive_multidimensional_ob_array_search_by_value($search_value, $array, $keys = array())
	{
		if (is_array($array) && count($array) > 0) {
			foreach ($array as $key => $value) {
				$temp_keys = $keys;

				// Adding current key to search path
				array_push($temp_keys, $key);

				// Check if this value is an array
				// with atleast one element
				if (is_array($value) && count($value) > 0) {
					$widget_type = isset($value['widgetType']) ? $value['widgetType'] : false;
					if ($widget_type) {
						if ($widget_type === $search_value) {
							$value['path'] = $temp_keys;
							return $value;
						} else {
							$res_path = self::recursive_multidimensional_ob_array_search_by_value(
								$search_value, $value['elements'], $temp_keys);
						}
						if ($res_path != null) {
							return $res_path;
						}
					} else {
						$res_path = self::recursive_multidimensional_ob_array_search_by_value(
							$search_value, $value['elements'], $temp_keys);
					}
					if ($res_path != null) {
						return $res_path;
					}
				}
			}
		}

		return null;
	}


	/**
	 * Check if checkout ajax or not
	 *
	 * @return bool
	 */
	public static function is_checkout_ajax() {
		if ( wp_doing_ajax() || isset( $_GET['wc-ajax'] ) ) {
			if ( isset( $_GET['wc-ajax'] ) && //phpcs:ignore
				isset( $_POST['_wcf_checkout_id'] ) //phpcs:ignore
			) {
				return true;
			}
		}

		return false;
	}


	/**
	 * Calculate discount price
	 *
	 * @param $discount_type
	 * @param $discount_value
	 * @param $product_price
	 * 
	 * @deprecated deprecated since 2.7.6
	 * 
	 * @return string
	 */
	public static function calculate_discount_price( $discount_type, $discount_value, $product_price ) {

		$custom_price = $product_price;
		if (!empty($discount_type)) {
			if ('discount-percentage' === $discount_type) {
				if ( $discount_value > 0 && $discount_value <= 100) {
					$custom_price = $product_price - (($product_price * $discount_value) / 100);
					$custom_price = number_format((float)$custom_price,2, '.', '');
				}
			} elseif ('discount-price' === $discount_type) {
				if ($discount_value > 0) {
					$custom_price = $product_price - $discount_value;
				}
			}
		}
		return number_format($custom_price, 2);
	}


	/**
	 * Calculate discount price
	 *
	 * @param $discount_type
	 * @param $discount_value
	 * @param $product_price
	 * 
	 * @return string
	 */
	public static function calculate_discount_amount( $discount_type, $discount_value, $product_price ) {
        
		if ( $discount_type !== 'original' && !empty($discount_type) ) {
			$discount = '';
			if ('discount-percentage' === $discount_type) {
				if ( $discount_value > 0 && $discount_value <= 100) {
					$discount = (($product_price * $discount_value) / 100);
				}
			} elseif ('discount-price' === $discount_type) {
				if ($discount_value > 0) {
					$discount = $discount_value;

				}
			}
			return number_format($discount, 2);
		}

		return false;
	}


	/**
	 * Get attributes for wpfunnels body wrapper
	 *
	 * @param string $template
	 * 
	 * @return string
	 */
	public static function get_template_container_atts( $template = '' ) {
		$attributes  = apply_filters( 'wpfunnels/page_container_atts', array() );
		$atts_string = '';
		foreach ( $attributes as $key => $value ) {
			if ( ! $value ) {
				continue;
			}
			if ( true === $value ) {
				$atts_string .= esc_html( $key ) . ' ';
			} else {
				$atts_string .= sprintf( '%s="%s" ', esc_html( $key ), esc_attr( $value ) );
			}
		}
		return $atts_string;
	}


	/**
	 * Get funnel link
	 * 
	 * @param $funnel_id
	 * 
	 * @return Mix
	 */
	public static function get_funnel_link( $funnel_id ) {
		if(!$funnel_id) {
			return;
		}
		$steps 		= self::get_steps($funnel_id);
		if( $steps && is_array($steps) ) {
			$first_step 	= reset($steps);
			$first_step_id 	=  $first_step['id'];
			return get_the_permalink($first_step_id);
		}
		return home_url();
	}


	/**
	 * Get wc fragment
	 *
	 * @param array $data
	 * 
	 * @return array
	 */
	public static function get_wc_fragments( $data = array() ) {
		ob_start();
		woocommerce_order_review();
		$woocommerce_order_review = ob_get_clean();

		ob_start();
		woocommerce_checkout_payment();
		$woocommerce_checkout_payment = ob_get_clean();

		$response = array(
			'cart_total'          => WC()->cart->total,
			'wc_custom_fragments' => 'yes',
			'fragments'           => apply_filters(
				'woocommerce_update_order_review_fragments',
				array(
					'.woocommerce-checkout-review-order-table' => $woocommerce_order_review,
					/*'.woocommerce-checkout-payment' => $woocommerce_checkout_payment,*/
				)
			),
		);

		if( !empty( $data ) ) {
			$response[ 'wpfunnels_data' ] = $data;
		}

		return $response;
	}


	/**
	 * Get supported builders
	 *
	 * @return array
	 */
	public static function get_supported_builders() {
		$builders = array(
			'elementor' 	=> 'Elementor',
			'gutenberg' 	=> 'Gutenberg',
			'divi-builder' 	=> 'Divi builder',
			'oxygen' 		=> 'Oxygen',
		);
		return apply_filters( 'wpfunnels/supported_builders', $builders );
	}

	/**
	 * Get custom url instead of deafult tahnkyou page
	 *
	 * @param mixed $step_id
	 *
	 * @return [type]
	 */
	public static function custom_url_for_thankyou_page( $step_id ){

		$isThankyou = self::check_if_this_is_step_type_by_id($step_id, 'thankyou');
		$isCustomUrl = get_post_meta( $step_id, '_wpfnl_thankyou_is_custom_redirect', true );
		$isDirect = get_post_meta( $step_id, '_wpfnl_thankyou_is_direct_redirect', true );

		if( $isThankyou && $isCustomUrl === 'on' ){
			$url = get_post_meta( $step_id, '_wpfnl_thankyou_custom_redirect_url', true );
			if( $url ){

				if( $isDirect === 'off' ){
					$redirectAfterSec = get_post_meta( $step_id, '_wpfnl_thankyou_set_time', true ) ? get_post_meta( $step_id, '_wpfnl_thankyou_set_time', true ) : 5;
					header( "refresh:".$redirectAfterSec.";url=".$url );
				}else{
					//unsell 'wpfunnels_automation_data' cookie and trigger 'wpfunnels/trigger_automation' hook
					self::unset_site_cookie( $step_id, 'wpfunnels_automation_data', 'wpfunnels/trigger_automation' );
					return $url;
				}
			}
			return false;
		}
		return false;
	}

	/**
	 * Check product has perfect variation or not
	 *
	 * @param $variation_id
	 * 
	 * @return Boolean
	 */
	public static function is_perfect_variations( $variation_id ){

		$blank_attr = [];
		$response = array(
			'status' => true,
			'data' => [],
		);
		$product = wc_get_product( $variation_id );
		if( $product ){
			$parent_id = $product->get_parent_id();
			$_product = wc_get_product( $parent_id );
			if( $_product ){
				$attrs = self::get_product_attr( $_product );
				$attributes = $product->get_attributes();
				if( !empty( $attributes) ){
					foreach($attributes as $attribute_key=>$attribute_value){

						if( !$attribute_value ){
							$blank_attr[$attribute_key] = $attrs[$attribute_key];
							$response['status'] = false;
						}
					}
				}
			}

		}
		$response['data'] = $blank_attr;
		return $response;
	}


	/**
	 * Get attributes of product
	 *
	 * @param Object $produt
	 */
	public static function get_product_attr( $product ){
		$attributes = $product->get_attributes();
		$attr_array = [];
		foreach($attributes as $attribute_key=>$attribute_value){

			$attribute_name = str_replace( 'attribute_', '', $attribute_key );
			$attr_value = $product->get_attribute( $attribute_name );
			$attr_value = strtolower($attr_value);

			if (strpos($attr_value, '|')) {
				$attr_array[$attribute_key] = explode("|",$attr_value);
			}else{
				$attr_array[$attribute_key] = explode(",",$attr_value);
			}
		}
		return $attr_array;

	}



	/**
	 * Remove site cookie
	 *
	 * @param $step_id, $cookie_name, $trigger_hook, $funnel_id
	 */
	public static function unset_site_cookie( $step_id, $cookie_name, $trigger_hook = '', $funnel_id = '' ){

		if( !$funnel_id ){
			$funnel_id = self::get_funnel_id_from_step( $step_id );
		}

		if( !$funnel_id ){
			return false;
		}

        $cookie             = isset( $_COOKIE[$cookie_name] ) ? json_decode( wp_unslash( $_COOKIE[$cookie_name] ), true ) : array();
        if(!isset($cookie['funnel_id'])) {
            $cookie['funnel_id']   = $funnel_id;
        }
        $cookie['funnel_status']   = 'successful';

		if(isset( $_COOKIE[$cookie_name] )){
			if( $trigger_hook ){
				do_action( $trigger_hook, $cookie );
			}
        }
		// unset cookie
		ob_start();
		if( isset( $_COOKIE[$cookie_name] ) ){
			setcookie( $cookie_name, null, strtotime( '-1 days' ), '/', COOKIE_DOMAIN );
		}
		ob_end_flush();
	}


	/**
     * Get formated product name
     *
	 * @param Object $product
	 * 
     * @return String
     */
    public static function get_formated_product_name( $product, $formatted_attr = [] ){
        $_product     = wc_get_product( $product );
		$each_child_attr = [];
		$_title = '';
		if( $_product ){
			if( !$formatted_attr ){
				if( 'variable' === $_product->get_type()  || 'variation' === $_product->get_type() || 'subscription_variation' === $_product->get_type() || 'variable-subscription' === $_product->get_type() ){
					$attr_summary = $_product->get_attribute_summary();
					$attr_array   = explode( ",", $attr_summary );
					
					foreach ( $attr_array as $ata ) {
						$attr              = strpbrk( $ata, ":" );
						$each_child_attr[] = $attr;
					}
				}

			}else{
				foreach ( $formatted_attr as $attr ) {
					$each_child_attr[] = ucfirst($attr);
				}
			}
			if( $each_child_attr ){
				$each_child_attr_two = [];
				foreach ( $each_child_attr as $eca ) {
					$each_child_attr_two[] = str_replace( ": ", " ", $eca );
				}
				$_title = $_product->get_title() . " - ";
				$_title = $_title . implode( ', ', $each_child_attr_two );
			}else{
				$_title = $_product->get_title();
			}
		}

        return $_title;
    }


	/**
	 * Get page builder of a specific funnel by step Id from postmeta
	 *
	 * @param $funnel_id
	 * 
	 * @return String $builder_name
	 *
	 * @since 2.0.5
	*/
	public static function get_page_builder_by_step_id( $funnel_id ){
		$steps = self::get_steps( $funnel_id );
		$builder_name = '';
		if( isset($steps[0]) ){
			$first_step_id = $steps[0]['id'];
			// check builder is elementor or not
			$elementor_page = get_post_meta( $first_step_id, '_elementor_edit_mode', true );

			// check builder is divi or not
			$divi_page = get_post_meta( $first_step_id, '_et_pb_use_builder', true );

			//check Oxygen builder is not
			$oxygen_page = get_post_meta($first_step_id, 'ct_builder_shortcodes',true);

			if( $elementor_page ) {
				$builder_name = 'elementor';
			} elseif( 'on' === $divi_page ){
				$builder_name = 'divi-builder';
			} elseif (!empty($oxygen_page)){
				$builder_name = 'oxygen';
			} else {
				$builder_name = 'gutenberg';
			}
			if( $builder_name ){
				return $builder_name;
			}
		}
		return $builder_name;
	}


	/**
	 * Get checkout section heading settings
	 */
	public static function get_checkout_section_heading_settings( $type = '' , $step_id = '' ){

		if( self::is_pro_license_activated() ){
			if( $step_id ){
				$get_settings = get_post_meta( $step_id, '_wpfunnels_edit_field_additional_settings', true );
				if( !empty($get_settings) ){
					if( $type === 'billing' ){

						$settings = array(
							'custom_heading' => $get_settings['custom_billing_heading'],
						);
						return $settings;
					}
					if( $type === 'shipping' ){
						$settings = array(
							'custom_heading' => $get_settings['custom_shipping_heading'],
						);
						return $settings;
					}

					if( $type === 'order' ){
						$settings = array(
							'custom_heading' => $get_settings['custom_order_detail_heading'],
						);
						return $settings;
					}
				}

			}
		}
		return false;
	}

	/**
     * Check webhook addon activated or not
     *
     * @return Boolean
     */
    public static function is_webhook_activated(){
        if( is_plugin_active( 'wpfunnels-pro-webhook/wpfunnels-pro-webhook.php' )){
            return true;
        }
		return false;
    }


	/**
     * Check webhook addon license activated or not
     *
     * @return Boolean
     */
    public static function is_webhook_license_activated(){

        if( is_plugin_active( 'wpfunnels-pro/wpfnl-pro.php' )){
            return true;
        }
		return false;
    }


	/**
	 * Get gbf supported addons
	 */
	public static function get_supported_addons(){

		$addons = array(
			'global_funnel' => array(
				'features' => array(
					'trigger_options' => array(
						''         => __('Please select type..', 'wpfnl'),
						'category' => __('Product category is ..', 'wpfnl'),
						'product'  => __('Product is ..', 'wpfnl'),
					)
				)
			),
		);

		if( defined( 'WPFNL_PRO_GB_VERSION') && version_compare( WPFNL_PRO_GB_VERSION, '1.0.7', '>=')  ){
			$addons['global_funnel']['features']['trigger_options']['all_product'] = __('Any product is selected', 'wpfnl');
		}
		return apply_filters( 'wpfunnels/supported-addons', $addons );
	}

	/**
	 * Get supported GBF offer condition
	 */
	public static function get_supported_gbf_offer_condition(){
		$conditions = [
			'specificProduct' => 'Specific Product'
		];
		return apply_filters( 'wpfunnels/supported-gbf-offer-condition', $conditions );
	}


	/**
	 * Get product categories for GBF offer condition
	 */
	public static function get_categories_gbf_offer_condition(){
		$categories = array();
		if( self::is_wc_active() ){
			$orderby = 'name';
			$order = 'asc';
			$hide_empty = false ;
			$cat_args = array(
				'orderby'    => $orderby,
				'order'      => $order,
				'hide_empty' => $hide_empty,
			);
			$product_categories = get_terms( 'product_cat', $cat_args );
			if( !empty($product_categories) ){
				foreach($product_categories as $category){
					$categories[$category->slug] = $category->name;
				}
			}
		}
		return $categories;
	}


	public static function oxygen_builder_version_capability(){
		if (defined("CT_VERSION") && version_compare(CT_VERSION,'3.2','>=')){
			return true;
		}
		return false;
	}

	public static function get_funnel_utm_settings( $funnel_id = null ){
		if( $funnel_id ){
			$default_enable_settings = array(
				'utm_enable'	=> 'off',
				'utm_source' 	=> '',
				'utm_medium' 	=> '',
				'utm_campaign' 	=> '',
				'utm_content' 	=> '',
			);

			$utm_settings = get_post_meta( $funnel_id, '_wpfunnels_utm_params', true );
			return wp_parse_args($utm_settings, $default_enable_settings);
		}
		return false;
	}

	public static function get_funnel_settings( $funnel_id = null ){

		if( $funnel_id ){

			$utm_settings = self::get_funnel_utm_settings( $funnel_id );
			$is_fb_pixel = get_post_meta( $funnel_id, '_wpfunnels_disabled_fb_pixel', true ) ? get_post_meta( $funnel_id, '_wpfunnels_disabled_fb_pixel', true ) : 'no';
			$is_gtm = get_post_meta( $funnel_id, '_wpfunnels_disabled_gtm', true ) ? get_post_meta( $funnel_id, '_wpfunnels_disabled_gtm', true ) : 'no';

			return [
				'utm_settings' 	=> $utm_settings,
				'is_fb_pixel' 	=> $is_fb_pixel,
				'is_gtm' 		=> $is_gtm,
			];
		}
		return false;

	}


	/**
	 * Check discount is applicable or not in checkout page
	 */
	public static function discount_on_checkout( $step_id = null  ){
		if( $step_id ){
			$discount = get_post_meta( $step_id, '_wpfnl_checkout_discount_main_product', true );
			if( $discount ){
				return $discount;
			}
		}
		return false;
	}


	/**
	 * Calculate main product total amount in checkout page
	 *
	 * @param Array $checkout_product
	 * @param Object $cart_content
	 * @param String $step_id
	 *
	 * @return Mix
	 */
	public static function calculate_main_product_total( $checkout_product = null , $cart_content = null, $step_id = '' ){

		if( $checkout_product && $cart_content ){
			$total = 0;
            
			foreach( $cart_content as $key=>$content ){
				if( isset($content['wpfnl_order_bump']) && $content['wpfnl_order_bump'] ){
					continue;
				}
				$product_id = '';
				if( !empty($content['variation_id']) ){
					$product_id = $content['variation_id'];
				}

				if( !$product_id ){
					$product_id = isset($content['product_id']) ? $content['product_id'] : '';
				}
				
				$checkout_product = maybe_unserialize( $checkout_product );
				$key = array_search($product_id, array_column($checkout_product, 'id'));
				if( $key !== false ){
					if (wc_tax_enabled()) {
						if ( !wc_prices_include_tax() ) {
							$content['line_total'] = number_format($content['line_total'], 2);
							$content['line_total'] = filter_var($content['line_total'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
							$total = $total + $content['line_total'];
						}else{

							$content['line_total'] = number_format($content['line_total'], 2);
							$content['line_total'] = filter_var($content['line_total'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
							$total = $total + $content['line_total'] + $content['line_tax'];

						}
					}else{
						$content['line_total'] = number_format($content['line_total'], 2);
						$content['line_total'] = filter_var($content['line_total'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
						$total = $total + $content['line_total'] + $content['line_tax'];
					}
				}
			}

			$total = number_format($total, 2);
			$total = filter_var($total, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
			return $total;
		}
		return false;
	}



	/**
	 * Calculate cart total amount in checkout page
	 *
	 * @param Array $checkout_product
	 * @param Object $cart_content
	 * @param String $step_id
	 *
	 * @return bool|int|float
	 */
	public static function calculate_cart_total( $checkout_product = null , $cart_content = null, $step_id = '' ){

		if( $checkout_product && $cart_content ){
			$total = 0;

			foreach( $cart_content as $key=>$content ){
				$product_id = $content['variation_id'] ? $content['variation_id'] : $content['product_id'];

				if (wc_tax_enabled()) {
					if ( !wc_prices_include_tax() ) {
						$content['line_total'] = number_format($content['line_total'], 2);
						$content['line_total'] = filter_var($content['line_total'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
						$total = $total + $content['line_total'];
					}else{
						$content['line_total'] = number_format($content['line_total'], 2);
						$content['line_total'] = filter_var($content['line_total'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
						$total = $total + $content['line_total'] + $content['line_tax'];

					}
				}else{
					$content['line_total'] = number_format($content['line_total'], 2);
					$content['line_total'] = filter_var($content['line_total'], FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
					$total = $total + $content['line_total'] + $content['line_tax'];

				}
			}
			$total = number_format($total, 2);
			$total = filter_var($total, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
			return $total;
		}
		return false;
	}



	/**
	 * Check array type ( multi-dimentional or one dimentional )
	 *
	 * @param Array
	 * 
	 * @return Bool
	 */
	public static function check_array_is_multidimentional( $multi_array = null ){

		if( $multi_array ){
			foreach($multi_array as $array){
				if(is_array($array)){
					return true;
				}else{
					return false;
				}
			}
			return true;
		}
		return false;
	}


	/**
	 * Migrate old Order bump settings with multiple order bump
	 */
	public static function migrate_order_bump( $prev_settings = null, $step_id = null ){
		if( $prev_settings && $step_id ){
			$multi_settings = array();
			if( !self::check_array_is_multidimentional( $prev_settings ) ){
				$multi_settings = array(
					$prev_settings
				);
				update_post_meta( $step_id, 'order-bump-settings', $multi_settings);
				return $multi_settings;
			}
		}
		return false;
	}


	/**
	 * N:B test case should be implemented
     * Get order bump settings by step id
     */
    public static function get_ob_settings( $step_id ){

        $all_settings = get_post_meta($step_id, 'order-bump-settings', true) ? get_post_meta($step_id, 'order-bump-settings', true) : array();
        $is_multiple = self::check_array_is_multidimentional( $all_settings );

        if( !$is_multiple && $all_settings ){
            $all_settings['name'] = 'Order bump';
            $all_settings = self::migrate_order_bump( $all_settings , $step_id );
        }

        if( is_array( $all_settings ) && count($all_settings) > 0 ) {
            foreach( $all_settings as $key=>$settings ){
                if( !isset($all_settings[$key]['discountOption']) && 'original' !== $all_settings[$key]['discountOption'] ) {
                    $product	= wc_get_product( $all_settings[$key]['product'] );
					if( $product ){
						if( $product->is_on_sale() ) {
							$price = wc_format_sale_price( $product->get_regular_price() * $all_settings[$key]['quantity'], $product->get_sale_price() ? $product->get_sale_price() * $all_settings[$key]['quantity'] :  $product->get_regular_price() * $all_settings[$key]['quantity']);
						} else {
							$price = $product->get_regular_price();
							if( is_plugin_active( 'woocommerce-subscriptions/woocommerce-subscriptions.php' ) ){
								$signUpFee 	= \WC_Subscriptions_Product::get_sign_up_fee( $product );
								$price 		=	$price + $signUpFee;
							}
							$price = wc_price( $price * $all_settings[$key]['quantity'] );
						}
						$all_settings[$key]['htmlPrice']	= $price ? $price : $all_settings[$key]['htmlPrice'];
					}
                }
            }
        	return $all_settings;
		}

        return [];
    }


	/**
	 * Check if funnel canvas or not
	 *
	 * @return bool
	 * @since  2.3.8
	 */
	public static function is_funnel_canvas_window() {
		if( is_admin() && isset( $_GET['page'] ) && 'edit_funnel' === $_GET['page'] ) {
			return true;
		}
		return false;
	}


	/**
     * Get all funnel from post
     *
	 * @return $funnels
	 * @since  2.4.4
     */
    public static function get_all_funnels( $status = 'publish' ){
        
		if( 'all' == $status ){
			$arg = [
				'post_type'   	=> WPFNL_FUNNELS_POST_TYPE,
				'fields'        => 'ID',
				"orderby" 		=> "date",
				"order" 		=> 'ASC',
				'numberposts'   => -1
			];
		}else{
			$arg = [
				'post_type'   	=> WPFNL_FUNNELS_POST_TYPE,
				'post_status'   => $status,
				'fields'        => 'ID',
				"orderby" 		=> "date",
				"order" 		=> 'ASC',
				'numberposts'   => -1
			];
		}
		
        $funnels     = get_posts( $arg );
        return $funnels;
    }


	/**
     * Get all funnel from post
     *
	 * @return $funnels
	 * @since  2.4.4
     */
    public static function get_all_steps(){

        $steps     = get_posts(
            array(
                'post_type'     => 'wpfunnel_steps',
                'post_status'   => 'publish',
                'fields'        => 'ID',
                "orderby" => "date",
                "order" => 'ASC',
            )
        );
        return $steps;
    }


	/**
     * Add type meta for each funnel
     *
     * @since 2.4.4
     */
    public static function add_type_meta(){

		$is_added = get_option( '_wpfnl_added_type_meta' );
		if( !$is_added ){
			update_option( '_wpfnl_added_type_meta', 'yes' );
			$funnels = self::get_all_funnels();
			foreach($funnels as $funnel){
				$type = get_post_meta( $funnel->ID, '_wpfnl_funnel_type', true );
				if( !$type ){
					update_post_meta( $funnel->ID, '_wpfnl_funnel_type', 'wc' );
				}
			}
		}

    }


	/**
	 * Enable lms settings
	 *
	 * @return Bool
	 * @since  2.4.5
	 */
	public static function is_enable_lms_settings(){
		$status  		= get_option( 'wpfunnels_pro_lms_license_status' );
        if( $status === 'active' ){
			return apply_filters('wpfunnels/is_enable_lms_settings', false);
		}
		return false;
	}

	/**
	 * Check if lms addon activated or not
	 *
	 * @return Bool
	 * @since  2.4.5
	 */
	public static function is_lms_addon_active(){

		if (in_array('wpfunnels-pro-lms/wpfunnels-pro-lms.php', apply_filters('active_plugins', get_option('active_plugins')))) {
			return true;
		}elseif( function_exists('is_plugin_active') ){
			if( is_plugin_active( 'wpfunnels-pro-lms/wpfunnels-pro-lms.php' ) ){
				return true;
			}
		}
		return false;
	}


	/**
	 * Get checkout tabs column name
	 *
	 * @return Array
	 * @since  2.4.5
	 */
	public static function get_checkout_columns( $step_id ){

		$funnel_id = get_post_meta( $step_id, '_funnel_id', true );
		$columns = [
			'product-name' 			=> 'Product',
			'product-price' 		=> 'Unit Price',
			'calculate-operator' 	=> '',
			'product-quantity' 		=> 'Quantity',
			'total-price' 			=> 'Total Price',
			'product-action' 		=> 'Actions',
		];

		return apply_filters('wpfunnels/checkout_columns', $columns, $funnel_id);
	}


	/**
	 * Supported order bump position
	 *
	 * @return Array
	 * @since  2.4.5
	 */
	public static function supported_orderbump_position( $step_id ){
		$funnel_id = get_post_meta( $step_id, '_funnel_id', true );

		$positions = [
			[
				'name'  => 'Before Order Details',
				'value' => 'before-order',
			],
			[
				'name'  => 'After Order Details',
				'value' => 'after-order',
			],
			[
				'name'  => 'Before Checkout Details',
				'value' => 'before-checkout',
			],
			[
				'name'  => 'After Customer Details',
				'value' => 'after-customer-details',
			],
			[
				'name'  => 'Before Payment Options',
				'value' => 'before-payment',
			],
			[
				'name'  => 'After Payment Options',
				'value' => 'after-payment',
			],
			[
				'name'  => 'Pop-up offer',
				'value' => 'popup',
			],
		];
		if( $funnel_id ){
			return apply_filters('wpfunnels/ob_positions', $positions, $funnel_id);
		}

		return $positions;

	}

	/**
	 * Get time zone in string
	 *
	 * @return String
	 * @since  2.4.6
	 */
	public static function wpfnl_timezone_string() {

		// If site timezone string exists, return it.
		$timezone = get_option( 'timezone_string' );
		if ( $timezone ) {
			return $timezone;
		}

		// Get UTC offset, if it isn't set then return UTC.
		$utc_offset = floatval( get_option( 'gmt_offset', 0 ) );
		if ( ! is_numeric( $utc_offset ) || 0.0 === $utc_offset ) {
			return 'UTC';
		}

		// Adjust UTC offset from hours to seconds.
		$utc_offset = (int) ( $utc_offset * 3600 );

		// Attempt to guess the timezone string from the UTC offset.
		$timezone = timezone_name_from_abbr( '', $utc_offset );
		if ( $timezone ) {
			return $timezone;
		}

		// Last try, guess timezone string manually.
		foreach ( timezone_abbreviations_list() as $abbr ) {
			foreach ( $abbr as $city ) {
				// WordPress restrict the use of date(), since it's affected by timezone settings, but in this case is just what we need to guess the correct timezone.
				if ( (bool) date( 'I' ) === (bool) $city['dst'] && $city['timezone_id'] && intval( $city['offset'] ) === $utc_offset ) { // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
					return $city['timezone_id'];
				}
			}
		}

		// Fallback to UTC.
		return 'UTC';
	}



	/**
	 * Get timezone offset in seconds.
	 *
	 * @since  2.4.6
	 * @return float
	 */
	public static function wpfnl_timezone_offset() {
		$timezone = get_option( 'timezone_string' );
		if ( $timezone ) {
			$timezone_object = new DateTimeZone( $timezone );
			return $timezone_object->getOffset( new DateTime( 'now' ) );
		} else {
			return floatval( get_option( 'gmt_offset', 0 ) ) * HOUR_IN_SECONDS;
		}
	}


	/**
	 * Check if integrations addon activated or not
	 *
	 * @return Bool
	 * @since  2.4.7
	 */
	public static function is_integrations_addon_active(){
		if (in_array('wpfunnels-pro-integrations/wpfunnels-pro-integrations.php', apply_filters('active_plugins', get_option('active_plugins')))) {
			return true;
		}elseif( function_exists('is_plugin_active') ){
			if( is_plugin_active( 'wpfunnels-pro-integrations/wpfunnels-pro-integrations.php' ) ){
				return true;
			}
		}
		return false;
	}


	/**
	 * Get template types
	 *
	 * @return array
	 * @since  2.4.11
	 */
	public static function get_template_types() {
		$general_settings = Wpfnl_functions::get_general_settings();
		if( self::is_wc_active() ){
			$types = array(
				array(
					'slug'   => 'wc',
					'label'  => 'Woo Templates',
				),
				array(
					'slug'   => 'lead',
					'label'  => 'Lead Gen Templates',
				)
			);
		}else{
			$types = array(
				array(
					'slug'   => 'lead',
					'label'  => 'Lead Gen Templates',
				)
			);
		}

		if( isset($general_settings['funnel_type']) && 'lead' === $general_settings['funnel_type'] ) {
			$types = array(
				array(
					'slug'   => 'lead',
					'label'  => 'Lead Gen Templates',
				)
			);
		}
		return apply_filters( 'wpfunnels/modify_template_type', $types );

	}


	/**
	 * Get global funnel type
	 *
	 * @return Mix
	 */
	public static function get_global_funnel_type(){
		$general_settings = self::get_general_settings();
		if( isset($general_settings['funnel_type']) ){
			return $general_settings['funnel_type'];
		}
		return false;
	}


	/**
	 * May be allow to create sales funnel
	 *
	 * @return Bool
	 */
	public static function maybe_allow_sales_funnel(){
		if( self::is_wc_active() || self::is_lms_addon_active() ){
			return true;
		}
		return false;
	}

	/**
	 * Retrieve all user role
	 *
	 * @return Array
	 */

	public static function get_all_user_roles(){
		global $wp_roles;

		$all_roles = isset($wp_roles->roles) ? $wp_roles->roles : [];
		$editable_roles = apply_filters('editable_roles', $all_roles);

		return $editable_roles;
	}


	/**
	 * Retrive all the product tags
	 *
	 * @return array
	 * @since  2.4.11
	 */
	public static function get_all_tags(){
		$term_array = array();
		if( self::is_wc_active() ){
			$terms = get_terms( 'product_tag' );
			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ){
				foreach ( $terms as $term ) {

					$term_array[] = [
						'id' => isset($term->term_id) ? $term->term_id : "",
						'name' => isset($term->name) ? $term->name : "",
					];
				}
			}
		}
		return $term_array;
	}

	/**
     * Get all shiiping classes
     *
     * @return array
     * @since  2.4.10
     */
    public static function get_shipping_classes(){

		if( self::is_wc_active() ){
			$shipping_classes = WC()->shipping()->get_shipping_classes();
			if( $shipping_classes ){
				return $shipping_classes;
			}
		}
		return [];
    }


	/**
	 * Maybe global funnel
	 *
	 * @param String
	 * 
	 * @return String
	 *
	 * @since 2.4.14
	 */
	public static function maybe_global_funnel( $step_id = '' ){
		if( $step_id ){
			$funnel_id = get_post_meta($step_id, '_funnel_id', true);
			$is_gbf = get_post_meta( $funnel_id, 'is_global_funnel', true);
			if( $is_gbf ){
				return $is_gbf;
			}
		}
		return 'no';
	}


	/**
	 * Get Woocommerce billing/shipping countries
	 *
	 * @return array
	 * @since  2.4.18
	 */
	public static function get_countries(){
		if( self::is_wc_active() ){
			$wc_countries = new \WC_Countries();
			$countries = $wc_countries->get_countries();
			if( $countries ){
				return $countries;
			}
		}
		return [];
	}


	/**
	 * Check the checkout layout is express or not
	 *
	 * @return Bool
	 * @since  2.4.18
	 */
	public static function maybe_express_checkout( $checkout_id = ''){
		if( !$checkout_id ){
			global $post;
			if( !$post ){
				return false;
			}
			if ( wp_doing_ajax() || isset( $_GET['wc-ajax'] ) ) {
				$checkout_id =self::get_checkout_id_from_post($_POST);
			}else {
				$checkout_id = $post->ID;
			}
			if( !$checkout_id ){
				$checkout_id = get_the_ID();
			}
		}

		$layout = get_post_meta( $checkout_id, '_wpfnl_checkout_layout' , true);

		return 'wpfnl-express-checkout' === $layout;
	}



	/**
	 * Sanitize request data
	 * 
	 * @param $data
	 * 
	 * @return Mix
	 */
	public static function get_sanitized_get_post( $data = [] )
	{
		if ( is_array( $data ) && !empty( $data ) ) {
			return filter_var_array( $data, FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		}
		return array(
			'get' => filter_input_array(INPUT_GET, FILTER_SANITIZE_FULL_SPECIAL_CHARS),
			'post' => filter_input_array(INPUT_POST, FILTER_SANITIZE_FULL_SPECIAL_CHARS),
			'request' => filter_var_array( $_REQUEST, FILTER_SANITIZE_FULL_SPECIAL_CHARS),
		);
	}



	/**
	 * Update funnel type to lead
	 *
	 * @since 2.5.1
	 */
	public static function update_funnel_type_to_lead(){

		$saved_settings = self::get_admin_settings('_wpfunnels_general_settings');
		if( isset($saved_settings['funnel_type']) ){
			$saved_settings['funnel_type'] = 'lead';
			update_option( '_wpfunnels_general_settings', $saved_settings );
		}
	}


	/**
	 * Delete template transients
     *
	 * @since 2.5.1
	 */
	public static function wpfnl_delete_transient(){
		delete_option(WPFNL_TEMPLATES_OPTION_KEY.'_wc');
        delete_option(WPFNL_TEMPLATES_OPTION_KEY.'_lms');
        delete_option(WPFNL_TEMPLATES_OPTION_KEY.'_lead');

		delete_transient('wpfunnels_remote_template_data_wc_' . WPFNL_VERSION);
        delete_transient('wpfunnels_remote_template_data_lms_' . WPFNL_VERSION);
        delete_transient('wpfunnels_remote_template_data_lead_' . WPFNL_VERSION);
	}


	/**
	 * Get all the published funnel lists
	 * 
	 * @return Array $funnels 
	 * @since  2.5.6
	 */
	public static function get_funnel_list(){
		$funnels     = get_posts(
            array(
                'post_type'     => WPFNL_FUNNELS_POST_TYPE,
                'post_status'   => 'publish',
                'numberposts'   => -1,
                'fields'        => 'ID',
                "orderby" => "date",
                "order" => 'ASC',
            )
        );
		$formatted_funnel_data[' '] = __('Select funnel','wpfnl');
		if( is_array($funnels) ){
			foreach( $funnels as $funnel ){
				$is_gbf = get_post_meta( $funnel->ID, 'is_global_funnel', true );
				if( !$is_gbf || 'no' === $is_gbf ){
					$formatted_funnel_data[self::get_funnel_link( $funnel->ID )] = $funnel->post_title;
				}
			}
		}
        return $formatted_funnel_data;
	}

	/**
	 * Check the log status is enabled or not
	 * 
	 * @return Bool
	 * @since  2.5.9
	 */
	public static function maybe_logger_enabled(){
		$general_settings = get_option( '_wpfunnels_general_settings' );
		if( $general_settings && isset($general_settings['enable_log_status']) ){
			return 'on' === $general_settings['enable_log_status'];
		}
		return false;
	}
	

	/**
	 * Get supported step type
	 * Landing, Checkout, Thankyou steps are supported for free
	 * 
	 * @return Array $types
	 * @since  2.5.7
	 */
	public static function get_supported_step_type(){
		$types = [
			[
				'type' => 'landing',
				'name' => 'Landing',
			],
			[
				'type' => 'checkout',
				'name' => 'Checkout',
			],
			[
				'type' => 'thankyou',
				'name' => 'Thankyou',
			],
		];

		return apply_filters( 'wpfunnels/supported_step_type', $types );
	}


	/**
	 * Check user permission to allow WPFunnels
	 * 
	 * @param Array $permitted_role
	 * 
	 * @return String $role
	 * @since  2.6.2
	 */
	public static function role_permission_to_allow_wpfunnel( $permitted_role = [] ){

		return 'manage_options';
		$user = wp_get_current_user();
		if( isset($user->roles) ){
			$roles = ( array ) $user->roles;
			foreach( $roles as $role ){
				if( isset($permitted_role['allow_funnels'][$role]) ){
					return $role;
				}
			}
		}
		return 'administrator';
	}


	/**
	 * Check if mint crm is installed
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_mint_mrm_active()
	{
		return apply_filters( 'is_mail_mint_pro_active', false );
	}


	/**
     * Check if mint automation exist for a funnel by funnel id.
     *
     * @param int $funnel_id
     * @return bool
     * @since  1.0.0
     */
    public static function maybe_automation_exist_for_a_funnel( $funnel_id )
    {
        return apply_filters( 'wpfunnels/maybe_automation_exist_for_a_funnel', false, $funnel_id );
    }

	/**
	 * Supported steps for allow settings
	 * 
	 * @return array
	 * @since  2.6.6
	 */
	public static function get_settings_steps(){
		$settings_steps  =[
			'upsell',
			'downsell',
			'checkout',
			'thankyou',
			'delay',
			'sendMail',
			'createUser',
			'addTag',
			'addList',
			'removeTag',
			'removeList',
			'wpf_optin_submit',
			'wpf_order_placed',
			'wpf_cta_triggered',
			'wpf_order_bump_accepted',
			'wpf_offer_trigger',
		];

		return apply_filters( 'wpfunnels/supported_settings_steps', $settings_steps );
	}


	/**
	 * Supported steps for allow only settings
	 * 
	 * @return array
	 * @since  2.6.6
	 */
	public static function get_only_settings_steps(){
		$settings_steps  =[
			'delay',
			'sendMail',
			'createUser',
			'addTag',
			'addList',
			'removeTag',
			'removeList',
			'wpf_optin_submit',
			'wpf_order_placed',
			'wpf_cta_triggered',
			'wpf_order_bump_accepted',
			'wpf_offer_trigger',
		];

		return apply_filters( 'wpfunnels/supported_only_settings_steps', $settings_steps );
	}


	/**
	 * 
	 */
	public static function get_default_automation(){
		$automation_data = [
			'name'      		=> 'automation-1',
			'status'    		=> 'active',
			'author'    		=> 1,
			'trigger_name'    	=> 'wpf-offer-trigger',
			'steps'     => [
				[
						'step_id' => 'mg3e1i',
						'key' => 'wp_user_login',
						'type' => 'trigger',
						'settings' => [
		
						],
						'next_step_id' => 'zytv4'
				],
				[
						'step_id' => 'zytv4',
						'key' => 'createUser',
						'type' => 'action',
						'settings' => [
		
						],
						'next_step_id' => '9w8vjh'
				],
				[
						'step_id' => '9w8vjh',
						'key' => 'addTag',
						'type' => 'action',
						'settings' => [
							'tag_settings' => [
								'tags' => [
									[
										'id'    => 3,
										'title' => 'demo'                              
									]
								]
							]
						],
						'next_step_id' => ''
				],
			],
			'created_ago' => '3 seconds'
		];
		
		return $automation_data;
	}


	/**
	 * Get mint triggers
	 * 
	 * @return array
	 */
	public static function get_mint_triggers(){
		$triggers = [
			'wpf_optin_submit',
			'wpf_order_placed',
			'wpf_cta_triggered',
			'wpf_order_bump_accepted',
			'wpf_offer_trigger',
		];

		return apply_filters( 'wpfunnels/mint_triggers', $triggers );
	}


	/**
	 * Get mint actions
	 * 
	 * @return array
	 */
	public static function get_mint_actions(){
		$actions = [
			'delay',
			'sendMail',
			'createUser',
			'addTag',
			'addList',
			'removeTag',
			'removeList',
		];

		return apply_filters( 'wpfunnels/mint_actions', $actions );
	}


	/**
	 * Get all tags/list from mint
	 * 
	 * @return array
	 */
	public static function get_mint_contact_groups( $type = 'tags' ){
		$class_name = "Mint\\MRM\\DataBase\\Models\\ContactGroupModel";
		if( class_exists($class_name) ){
			
			$groups = $class_name::get_all_lists_or_tags($type);
			if( $groups && is_array($groups) ){
				$formatted_groups = [];
				$i=1;
				foreach( $groups as $group ){
					$data = [
						'index' 	=> $i,
						'title' 	=> $group['title'],
						'id' 		=> $group['id'],
						'checked' 	=> false,
					];
					array_push( $formatted_groups, $data );
					$i++;
				}
				
				return $formatted_groups;
			}
		}
		return [];
	}


	/**
	 * Get selected steps
	 * 
	 * @param string $type
	 * @param int $funnel_id
	 * 
	 * @return array
	 */
	public static function get_selected_steps( $type, $funnel_id ){
		if( $funnel_id ){
			$steps = get_post_meta( $funnel_id, '_steps', true );
			if( $steps && is_array($steps) ){
				$i = 1;
				$formatted_steps = [];
				$step_types = [$type];
				if( 'landing' == $type ){
					$step_types = [ $type, 'custom' ];
				}
				foreach($steps as $step ){
					if( in_array($step['step_type'], $step_types) ){
						$data = [
							'id' 	=> $i,
							'title' => $step['name'],
							'value' => $step['id'],
						];
						array_push( $formatted_steps, $data );
						$i++;
					}
				}
				
				return $formatted_steps;
			}
		}
		return [];
	}


	/**
	 * Get all order bump of a funnel
	 * 
	 * @param int $funnel_id
	 * 
	 * @return array
	 */
	public static function get_all_ob( $funnel_id ){
		if( $funnel_id ){
			$steps = self::get_steps( $funnel_id );
			if( $steps && is_array($steps) ){
				$key = array_search('checkout', array_column($steps, 'step_type'));
				if( false !== $key ){
					if( isset( $steps[$key]['id'] ) ){
						$checkout_step_id =  $steps[$key]['id'];
						$order_bumps = get_post_meta( $checkout_step_id, 'order-bump-settings', true);
						if( $order_bumps && is_array($order_bumps) ){
							$i = 1;
							$formatted_obs = [];
							foreach($order_bumps as $index=>$ob ){
								$data = [
									'id' 	=> $i,
									'title' => $ob['name'],
									'value' => $index,
								];
								array_push( $formatted_obs, $data );
								$i++;
							}
							return $formatted_obs;
						}
					}
				}
			}
		}
		return [];
	}


	/**
	 * Get sequences from campaign table
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public static function get_sequences() {
		if( self::is_mint_mrm_active() && class_exists("Mint\\MRM\\DataBase\\Tables\\CampaignSchema") ){
			
			global $wpdb;
			$class_name = "Mint\\MRM\\DataBase\\Tables\\CampaignSchema";
			$campaign_table = $wpdb->prefix . $class_name::$campaign_table;
			$results = $wpdb->get_results( $wpdb->prepare( "SELECT id as value, title as label FROM $campaign_table WHERE type = %s AND status = %s ", 'automation', 'created' ), ARRAY_A ); // phpcs:ignore.
			$default        = array(
				array(
					'value' => '',
					'label' => 'Select Sequence',
				),
			);
			if( !empty($results) ){
				return array_merge( $default, $results );
			}else{
				return $default;
			}
			
		}
		return [];
	}


	/**
	 * Get CFE deafult fields
	 * 
	 * @return array
	 * @since 2.7.0
	 */
	public static function get_cfe_default_fields(){
		if( self::is_wc_active() ){
			$fields = [];
			$billing_default_field = WC()->countries->get_address_fields();
			foreach($billing_default_field as $key => $sf){
				$billing_default_field[$key]['enable'] = 1;
				$billing_default_field[$key]['show'] = 1;
				$billing_default_field[$key]['delete'] = 0;
				if(!(isset($sf['type']))){
					$billing_default_field[$key]['type'] = null;
				}
				if(!(isset($sf['placeholder']))){
					$billing_default_field[$key]['placeholder'] = null;
				}
				if(!(isset($sf['label']))){
					$billing_default_field[$key]['label'] = null;
				}
				if(!(isset($sf['validate']))){
					$billing_default_field[$key]['validate'] = null;
				}
				if(!(isset($sf['default']))){
					$billing_default_field[$key]['default'] = null;
				}
			}

			$countries = new \WC_Countries();
			$shipping_default_fields = $countries->get_address_fields( $countries->get_base_country(),'shipping_');
			foreach($shipping_default_fields as $key => $sf){
				$shipping_default_fields[$key]['enable'] = 1;
				$shipping_default_fields[$key]['show'] = 1;
				$shipping_default_fields[$key]['delete'] = 0;
				if(!(isset($sf['type']))){
					$shipping_default_fields[$key]['type'] = null;
				}
				if(!(isset($sf['placeholder']))){
					$shipping_default_fields[$key]['placeholder'] = null;
				}
				if(!(isset($sf['label']))){
					$shipping_default_fields[$key]['label'] = null;
				}
				if(!(isset($sf['validate']))){
					$shipping_default_fields[$key]['validate'] = null;
				}
				if(!(isset($sf['default']))){
					$shipping_default_fields[$key]['default'] = null;
				}
			}

			$order_comment = array(
				'type'        => 'textarea',
				'class'       => array( 'notes' ),
				'label'       => __( 'Order notes', 'woocommerce' ),
				'placeholder' => esc_attr__(
								'Notes about your order, e.g. special notes for delivery.',
								'woocommerce'),
				'name' 		  => 'order_comments',
				'required'    => false,
				'enable'      => 1,
				'show'        => 1,
				'default'     => null,
				'validate'    => null,
				'delete'      => 0,
			);
			
			$additional_default_fields = array(
				'order_comments' => $order_comment
			);

			$fields = [
				'billing'  		=> $billing_default_field, 	
				'shipping' 		=> $shipping_default_fields,
				'additional'  	=> $additional_default_fields
			];
			return $fields;
		}
	} 

	/**
	 * Generates and updates the first step id of the funnel
	 * 
	 * @param int|string $funnel_id Funnel id.
	 * @since 2.8.0
	 */
    public static function generate_first_step( $funnel_id, $steps = [] ) {
        if( $funnel_id ) {
            $steps = !empty( $steps ) ? $steps : get_post_meta( $funnel_id, '_steps', true ); // Get step order

            $priority_order = [ 'landing', 'custom', 'upsell', 'downsell', 'checkout', 'thankyou' ];
            $first_step     = [ 'id' => null, 'step_type' => '' ];

            if( is_array( $steps ) && !empty( $steps ) ) {
                foreach( $steps as $step ) {
                    if(
                        isset( $step[ 'step_type' ], $step[ 'id' ] )
                        && ( !$first_step[ 'id' ] || ( array_search( $step[ 'step_type' ], $priority_order ) < array_search( $first_step[ 'step_type' ], $priority_order ) ) )
                    ) {
                        $first_step[ 'id' ]        = $step[ 'id' ];
                        $first_step[ 'step_type' ] = $step[ 'step_type' ];
                    }
                }
                self::update_funnel_first_step( $funnel_id, $first_step[ 'id' ] );
            }
        }
    }

    /**
     * Get the first step of a funnel
     *
     * @param int|string $funnel_id Funnel id.
     * @return string|int
     * @since 2.8.0
     */
    public static function get_first_step( $funnel_id ) {
        return get_post_meta( $funnel_id, '_first_step', true );
    }

    /**
     * Updates the first step id of the funnel
     *
     * @param int|string $funnel_id Funnel id.
     * @param int|string $first_step_id First step id of the funnel.
     * @return void
     */
    public static function update_funnel_first_step( $funnel_id, $first_step_id ) {
        update_post_meta( $funnel_id, '_first_step', $first_step_id );
    }


	/**
	 * Mail mint email
	 * 
	 * @return mix
	 */
	public static function get_mailmint_email(){
		if( self::is_mint_mrm_active() ){
			$email_settings = get_option( '_mrm_email_settings' );
			if( isset( $email_settings['from_email'] ) ){
				return $email_settings['from_email'];
			}
		}
		return false;
	}


	/**
	 * Maybe duplicate step
	 * 
	 * @param $step_id Step ID
	 * 
	 * @since 2.7.7
	 * @return bool
	 */
	public static function maybe_duplicate_step( $step_id ){
		
		if( $step_id ){
			$is_duplicate = get_post_meta( $step_id, '_is_duplicate', true );
			if( 'yes' === $is_duplicate ){
				return true;
			}
		}	
		return false;
	}


	/**
	 * Retrieves all created coupons from WooCommerce.
	 * This function fetches all coupon posts of the 'shop_coupon' post type that are currently published in WooCommerce.
	 * 
	 * @since 2.7.9
	 * @return array An array of coupon codes and their corresponding names.
	 */
	public static function get_all_coupons(){
		global $wpdb;
		$table_name = $wpdb->prefix . 'posts';
        $coupon_type = 'shop_coupon';
        $post_status = 'publish';
        $query = $wpdb->prepare(
            "SELECT post_title FROM $table_name WHERE post_type = %s AND post_status = %s ORDER BY post_name ASC LIMIT 100",
            $coupon_type,
            $post_status
        );
		$coupon_codes = [];
        $coupons = $wpdb->get_col($query);
		if( is_array( $coupons ) ){
			foreach ( $coupons as $coupon ) {
				$coupon_codes[] = [
					'name'	=> $coupon,
				];
			}
		}
		return $coupon_codes;
	}


	/**
	 * Retrieves the applied coupon for a given step ID.
	 * This function checks if a step ID is provided and retrieves the applied coupon associated with that step.
	 * If no step ID is provided or the applied coupon is not found, it returns false.
	 * 
	 * @param int $step_id The ID of the step for which to retrieve the applied coupon.
	 * 
	 * @since 2.7.9
	 * @return string|false The selected coupon code if found and auto-coupon feature is 
	 */
	public static function get_applied_coupon( $step_id ){
		if( !$step_id ){
			return false;
		}
		$auto_coupon 		=  get_post_meta($step_id, '_wpfnl_checkout_auto_coupon', true);
		if( isset( $auto_coupon['enableAutoCoupon'],  $auto_coupon['selectedCoupon'] ) &&  'yes' === $auto_coupon['enableAutoCoupon'] && $auto_coupon['selectedCoupon'] ){
			return $auto_coupon['selectedCoupon'];
		}
		return false;
	}

	/**
	 * Get step IDs for a given funnel.
	 *
	 * Retrieves an array of step IDs associated with a specific funnel, including A/B testing variations.
	 *
	 * @param int $funnel_id The ID of the funnel.
	 * @return array An array of step IDs.
	 * @since 2.7.10
	 */
	public static function get_step_ids( $funnel_id ){
		$steps = self::get_steps( $funnel_id );
        
        $all_steps_including_ab = array();
        $all_steps_including_ab_ids = array();
		if ( is_array($steps) ){
			foreach ( $steps as $step ) {
				if ( isset( $step['id'] ) ){
					$all_steps_including_ab = get_post_meta( $step['id'], '_wpfnl_ab_testing_start_settings' ,true );
					$ids = array_column($all_steps_including_ab['variations'], 'id');
					$all_steps_including_ab_ids = array_merge($all_steps_including_ab_ids, $ids);
				}
       	 	}
		}
        
        return $all_steps_including_ab_ids;
	}

	/**
	* Validates the order owner's authorization.
	*
	*
	* @param WC_Order $order The order object to validate.
	* @return bool
	* @since 2.7.13
	*/
	public static function is_valid_order_owner( $order ){
		
		if (!self::is_wc_active() || false === is_a( $order, 'WC_Order' )){
			return false;
		}

		// If it's not a guest order, current user can pay but only if it's their own order,
		// or they are an admin or shop manager or they are in a same session(for guest).

		$order_has_customer = ! empty( $order->get_customer_id() );
		$is_authorized_owner = false;

		$has_owner = !empty ( WC()->session->get('wpfnl_order_owner') );

		if ( $order_has_customer ) {
			$is_order_owned_by_current_user = $order->get_customer_id() === get_current_user_id();

			$is_authorized_owner = $is_order_owned_by_current_user || current_user_can( 'manage_woocommerce' );
		}

		if (!$is_authorized_owner && !$has_owner){
			return false;
		}
		return true;
	}
}
